### ### ### ### ### ### ### ### ### ###
### Option implied Hurst exponent   ###
### ### ### ### ### ### ### ### ### ###

# Description -------------------------------------------------------------

# This code consists of 4 sections:
#     (I)   computation of option-implied Hurst exponents
#     (II)  computation of historical Hurst exponents
#     (III) surrogate analysis: S&P 500 as if there was a constant Hurst exponent
#     (IV)  investment examples

# Initialization
library(pracma)
source("functions.R")



# I. Compute implied Hurst ------------------------------------------------

# Input Data:
load("mfiv.RData")
# This loads a data.frame called "mfiv". It represents daily simple variance swaps
# of the S&P 500 index with maturities of 1, 3, 6, 9 and 12 months.
# The data is provided by Grigory Vilkov and can be found under the doi link:
# https://doi.org/10.17605/OSF.IO/Z2486
# The input file consists of 5 columns and 6293 rows:  
# Each row represents a day, 1996-01-04 to 2020-12-31.
# Each column a time-to-maturity.

time <- as.Date(rownames(mfiv))     # days for which we have observations
tau <- c(30,91,182,273,365)/365     # time to maturity
F2 <- as.matrix(mfiv) %*% diag(tau) # de-annualize variance
y <- log(F2)                        # log variance: dependent variable
x <- log(tau)                       # log time to maturity: explanatory variable
mdl <- lm.fit(cbind(1,x), t(y))     # performs the fit for each row
H <- mdl$coefficients[2,]/2         # estimated implied Hurst exponents over time
rho_imp <- 2*H-1                    # transformed Hurst exponent

plot(time, H, type="l", main="implied Hurst exponent"); grid()


# II. Compute realized Hurst ----------------------------------------------

# Input Data:
load("sp500.RData") 
# This loads a vector "r" of daily log-returns of the S&P 500 index, computed from adjusted closing prices.
# Data source: Refintiv Datastream. Authors do not have permission to share the data.

n <- length(r)
sc <- logseq(10, n/10, 34)
mdl <- mfdfa(r, sc, overlap = T)
rH <- local_H(mdl, align="left")
rH <- rH$Ht
rho_hist <- 2*rH-1

plot(time, rho_imp, type="l", main="implied vs. future-realized Rho"); grid()
lines(time, rho_hist, col=2)


# III. S&P 500 without multi-fractality -----------------------------------

rs <- iaaft(r, 10^3)
md <- apply(rs, 2, max_dd)
md0 <- max_dd(r)
cat("Maximum Drawdown: \n original: ", -md0, "\n", "w/o multifrac.: ", -median(md))

# IV. Investment Examples -------------------------------------------------

rm(list=setdiff(ls(), c("rho_imp", "rho_hist", "n"))) # clear work space
load("spy_rf.RData") 
# This loads a data.frame called "spy_rf" holding daily prices of the SPY and the 1-month T-Bill rate (RF). 
# Authors do not have permission to share the data.

rf <- (spy_rf$RF[-1]+1)^(1/360)-1
r <- diff(log(spy_rf$SPDR_SP500_ETF))
r_discr <- exp(r)-1

# sentiment bet, PF1:
signal <- as.numeric(rho_imp > 0.1)
signal[which(rho_imp >= 0 & signal <=0.1)] <- 0.5
w1 <- signal[-n]
r_pf1_discr <- w1*r_discr + (1-w1)*rf
r_pf1 <- log(r_pf1_discr + 1)

# correlation bet, PF2:
rho_P <- rho_imp - cumsum(rho_imp-rho_hist)/(1:n)
signal <- as.integer(sign(rho_P) == sign(c(NA, r)))
w2 <- signal[-n]
w2[1] <- 0 
r_pf2_discr <- w2*r_discr + (1-w2)*rf
r_pf2 <- log(r_pf2_discr + 1)

plot(cumsum(r), type="l"); grid()
lines(cumsum(r_pf1), col=2)
lines(cumsum(r_pf2), col=4)

# Sharpe Ratio:
rfl <- log(rf+1)
mean(r - rfl)/sd(r)*sqrt(252)
mean(r_pf1 - rfl)/sd(r_pf1)*sqrt(252)
mean(r_pf2 - rfl,na.rm=T)/sd(r_pf2,na.rm=T)*sqrt(252)

