%% Code for application of SFS in DFM to study global macro-financial cycles
% Sebastian Hienzsch, Uni Gttingen
% date: 08.07.2024

%% Preample
disp('*** SFS-DFM Analysis ****');
disp(' ')

%% Main settings of interest
disp('Preliminary setup...');
% Choice of financial variable to include in model with realDGP
variable_option = 1; % 1: Capital inflows       -> baseline
                     % 2: Capital outflows
                     % 3: Hot capital inflows
                     % 4: Hot capital outflows
                     % 5: Credit                -> baseline
                     % 6: Bank loans

financial_variables = {'Capital inflows','Capital outflows','Hot capital inflows', 'Hot capital outflows', 'Credit', 'Loans'};

disp('************************')
disp(string(append('RUN MODEL with rGDP and ',financial_variables(variable_option))));
disp('************************')

% Choice which factors to include
model_option = 1; % 1: runs SFS
                  % 2: includes all factors (GMF)
                  % 3: gives baseline results for model preferred by SFS, i.e. includes only G and F (Baseline_GF)

if model_option == 1
    model_est = [2 2];   % when estimating: 0 for excluding factor, 1 for including, 2 for sampling indicator
                         % first input is for global, second for variable-specific groups and third for country
    model = 'SFS';
elseif model_option == 2
    model_est = [1 1];   
                         
    model = 'GMF'; 
elseif model_option == 3
    model_est = [1 1];  
                         
    model = 'Baseline_GF'; 
else
    error('Wrong setting for model options!')
end

% Prior inclusion probbility
p0 = 0.5;

%% Other settings
rescale_option = 1; % 0: no, 1: AL, 2: trend GDP 3: standardize(unit variance) 

% Gibbs settings
b_in1  = 1500;            % burn-in where indicators are fixed
b_in2  = 1500;            % burn-in where indicators are sampled
b_in   = b_in1 + b_in2;   % full burn-in
ndraws = 30000;           % number of useful draws
it_print = 200;
thin   = 1;             % thin out, use every nth draw
lags   = 1;

% Choice normalization on loading global factor
ortho_est = 1; % 
norm = 0;      % choice for normalization on loading of global factor,
               % 0: sets mean to 1
disp('... done');

%% Data 
disp('Preparing data for analysis...');

% call data import and transfom function
y = gmfcy_data_OECD20(variable_option, rescale_option);

% Assumed number of factors in each layer
[T,I] = size(y); % time sample and total no. of data series
J = 2;      % no. of variables per country
N = I/J;    % no. of countries
Gnr = 1;    % no. of global common factors (i.e. global macro-fin fact G)
Fnr = 2;    % no. of global variable-specific factors (i.e. F^m F^f)
M = Gnr + Fnr; % no. of common factors
K = J/Fnr; % no. of sector-specific variables per country

% Indicators for country and sector
% macro (1) and financial (2) variable indicator k= {1,2}
ind_f = [ones(1,I/Fnr) 2*ones(1,I/Fnr)];

disp('... done');

%% Binary indicators
s_delta = ones(1,M);
if model_est(1,1) == 0; s_delta(1,1) = 0; end
if model_est(1,2) == 0; s_delta(2:1+Fnr) = zeros(1,Fnr); end 

% Binary indicator settings for common factors
model_exp              = zeros(1,1+Fnr);
model_exp(1,1)         = model_est(1,1); % global
model_exp(1,2:Fnr+1)   = model_est(1,2); % macro and fin

if model_option == 3 % run model preferred by SFS, i.e. includes only G and F (Baseline_GF)
    s_delta(1,2) = 0;
    
    model_exp              = zeros(1,1+Fnr);
    model_exp(1,1)         = model_est(1,1); % global
    model_exp(1,2)         = 0; % macro
    model_exp(1,3)         = model_est(1,2); % fin
end

%% Priors (hyperparameters)
% Normal
% standard deviations of factors
a0_std_fac = 0;                 A0_std_fac =  10;
% factor loadings
a0_lambda = 1;                  A0_lambda = 0.15^2;
% AR parameters
a0_ar = ones(lags,1)*0.5/lags;  A0_ar     = eye(lags)*0.15^2; 

% Inverse Gamma
% error variance of nu
bel_var_mu = 1;     % belief 
str_var_mu = 0.1;   % strength of belief   
c0 = T*str_var_mu;              C0 = c0*bel_var_mu;

%% Initialize 
% States
s_G = randn(T,Gnr)*0;   s_GG = [];
s_F = randn(T,Fnr)*0;   s_FF = [];
% Stdv states
s_sig_g = ones(Gnr,1); s_sig_gg = [];
s_sig_f = ones(Fnr,1); s_sig_ff = [];
s_sig_mu = ones(I,1);
% Loadings 
s_gam_g = ones(I,1);
s_gam_f = ones(I,1);
% AR Parameters
s_phi_g =  ones(Gnr,lags)*0.5./lags;
s_phi_f =  ones(Fnr,lags)*0.5./lags;
s_phi_mu = ones(I,lags)*0.5./lags;

%% Storage arrays
% store states
store_G = NaN(T,1,ndraws); store_F = NaN(T,Fnr,ndraws);
% store loadings
store_gam_g = NaN(I,ndraws); store_gam_f = NaN(I,ndraws);
% store stdv 
store_sig_g = NaN(Gnr,ndraws); store_sig_f = NaN(Fnr,ndraws); store_sig_mu = NaN(I,ndraws);
% store phi (AR)
store_phi_g = NaN(Gnr,ndraws); store_phi_f = NaN(Fnr,ndraws); store_phi_i = NaN(I,ndraws);
% store delta
store_delta = NaN(M,ndraws); store_p_delta = zeros(M,ndraws); 
% store VD
store_VD = zeros(I,3,ndraws);
% non-real numbers counter
count_nonreal = 0;

%% Gibbs sampling
tic;
disp('Starting up the Gibbs sampler...');
disp('Number of iterations');
timing_start = now;
h = waitbar(0,'simulation running');
for j=1:b_in+ndraws
    if mod(j,it_print) == 0
        disp(j);
    end    
    % ---------------------------------------------------------------------
    % Sequential sampling of the common factors from the conditional distribution $p(f|y,\beta)$
    % ---------------------------------------------------------------------
    factor_order = [2,1];
    for ii = 1:2
        switch factor_order(ii)
            case 1 
            % Variable-specific factors (f^m f^f)
            [s_FF, s_sig_ff] = states_F(y,...
                                        s_G,...
                                        s_phi_f, s_phi_mu,...
                                        s_sig_g, s_sig_f, s_sig_mu,...
                                        s_gam_g, s_gam_f,...
                                        ind_f, lags, ortho_est);
            % ************************
            real_states = isreal(s_FF);
            if real_states == 1
            s_F = s_FF;
            s_sig_f = s_sig_ff;
            elseif real_states == 0
            count_nonreal = count_nonreal + 1;
            end
            % ************************   
            case 2 
            % Global factor (g^mf)
            [s_GG, s_sig_gg] = states_G(y,...
                                        s_F,...
                                        s_phi_g, s_phi_mu,...
                                        s_sig_g, s_sig_f, s_sig_mu,...
                                        s_gam_g, s_gam_f,...
                                        ind_f, lags, ortho_est);                  
            % ************************
            real_states = isreal(s_GG);
            if real_states == 1
                s_G = s_GG;
                s_sig_g = s_sig_gg;
            elseif real_states == 0
                count_nonreal = count_nonreal + 1;
            end 
            % ************************
        end
    end
      
    % ---------------------------------------------------------------------
    % Sample the parameters \beta from the conditional distribution p(\beta|y,f)
    % ---------------------------------------------------------------------  
    
    % Stdv factors and binary indicators
    if j > b_in1 % burn-in where indicators are fixed
        s_indic = 1; 
    else
        s_indic = 0;
    end
    [s_sig_g,s_sig_f,...
        s_delta, s_p_delta] = para_stdv(y,s_G,s_F,...
                                       s_phi_mu,s_sig_mu,...
                                       a0_std_fac,A0_std_fac,...
                                       s_gam_g,s_gam_f,...
                                       ind_f,s_indic,...
                                       p0,s_delta,...
                                       model_exp);    
    
    % Loadings and var_mu
    [s_gam_g,s_gam_f, ...
        s_sig_mu,s_sig_g,s_sig_f] = para_loadings(y,s_G,s_F,...
                                                s_phi_mu,...
                                                s_sig_g,s_sig_f,...
                                                a0_lambda,A0_lambda,c0,C0,...
                                                ind_f,...
                                                model_est,norm);    
    
    % AR parameters of factors and idiosyncratic component
    [s_phi_g,s_phi_f,s_phi_mu] = para_ar(y,s_G,s_F,...
                                                a0_ar,A0_ar,...
                                                s_gam_g,s_gam_f,...
                                                s_sig_g,s_sig_f,s_sig_mu,...
                                                s_phi_g,s_phi_f,s_phi_mu,...
                                                ind_f,...
                                                model_est,lags);    
    
    
    % ---------------------------------------------------------------------
    % Variance decomposition
    % ---------------------------------------------------------------------

    VD = zeros(I,3);
    if lags == 1
        for i = 1:I
            total_var = s_gam_g(i,1)^2*s_sig_g^2/(1-s_phi_g^2)+...
                            s_gam_f(i,1)^2*s_sig_f(ind_f(1,i),1)^2/(1-s_phi_f(ind_f(1,i),1)^2)+...
                                    s_sig_mu(i,1)/(1-s_phi_mu(i,1))^2;
            VD(i,1)   = s_gam_g(i,1)^2*s_sig_g^2/(1-s_phi_g^2)/total_var;
            VD(i,2)   = s_gam_f(i,1)^2*s_sig_f(ind_f(1,i),1)^2/(1-s_phi_f(ind_f(1,i),1)^2)/total_var;
            VD(i,3)   = s_sig_mu(i,1)/(1-s_phi_mu(i,1))^2/total_var;
            
        end
    end
    
    % ---------------------------------------------------------------------
    % Store results
    % ---------------------------------------------------------------------
    if (j>b_in) && (mod(j,thin)==0)
        % loadings
        store_gam_g(:,j-b_in) = s_gam_g; 
        store_gam_f(:,j-b_in) = s_gam_f;
        % sign switch for stdv
        sign=1; if rand>0.5; sign =-1; end
        store_sig_g(:,j-b_in) = sign*s_sig_g;
        store_sig_f(:,j-b_in) = sign*s_sig_f;
        % store error variance
        store_sig_mu(:,j-b_in)= s_sig_mu.^2;
        % AR parameters
%         store_phi_i(:,j-b_in) = s_phi_mu;
%         store_phi_g(:,j-b_in) = s_phi_g;
%         store_phi_f(:,j-b_in) = s_phi_f;
        
        if size(s_G,1)>0
            store_G(:,j-b_in) = s_G*s_sig_g;
        end
        for f = 1:Fnr
            store_F(:,f,j-b_in) = s_F(:,f)*s_sig_f(f,:);
        end

        store_delta(:,j-b_in)   = s_delta(1,:)';
        store_p_delta(:,j-b_in) = s_p_delta(1,:)';
        
        store_VD(:,:,j-b_in)  = VD;
    end    
    
    % progress report
    if ~rem(j,10)
        waitbar(j/(b_in+ndraws), h, timing_message(j, b_in+ndraws, timing_start))
    end
end
toc;
close(h)
disp('... done');

%% Summarizing the results
disp('Summarize and store results...');
% summary of results
results.G  = summary(store_G);
results.F  = summary(store_F);
 
results.sig_g  = summary((store_sig_g));
results.sig_f  = summary((store_sig_f));

results.VD  = summary(store_VD);

VD_mean = results.VD.mean';

results.gam_g  = summary(store_gam_g);
results.gam_f  = summary(store_gam_f);
results.pdelta = mean(store_p_delta,2);
results.delta  = mean(store_delta,2);

% save results
filename = append('Est_results_GMFCy_','Model_',model,'_varopt_',financial_variables(variable_option),'_p0',num2str(p0),'.mat');
save(string(fullfile('results/',filename)),'results')

% Plot results
sim=0;
graphs(T,I,N,Fnr,ind_f,y,results,sim,model_est)
[mean(store_p_delta,2) mean(store_delta,2)]

if model_est(1,1)==2
    % store post factor inclusion probs
    tmp = results.pdelta';
    tmp_FactProbFileName = append('FactorProbResults','Model_',model,'_varopt_',financial_variables(variable_option),'_p0',num2str(p0),'.mat');
    save(string(fullfile('results/',tmp_FactProbFileName)),'tmp');
    % compute posterior model probabilities
    bin_mat=store_delta';
    perm_mat = dec2bin(0:(2^3)-1)-'0';
    [PM,dum,Pj] = unique(perm_mat,'rows') ;
    [dum, X] = ismember(bin_mat,PM,'rows') ; 
    N = histc(X,1:length(PM)) ;
    C = N(Pj) ;  
    [row,dum] = find(C);
    ii=length(row);
    mdl_freq=zeros(ii,5);
    for jj=1:ii
        mdl_freq(jj,:)=[perm_mat(row(jj,1),:) C(row(jj,1)) C(row(jj,1))./ndraws];
    end
    mprob = sortrows(mdl_freq,5); 
    head={'Global', 'Macro' ,'Fin',  'frequency', 'probability'};
    mdl_tbl=[head; num2cell(mprob)]
    tmp_FileName = append('ModelProbResults','Model_',model,'_varopt_',financial_variables(variable_option),'_p0',num2str(p0),'.mat');
    save(string(fullfile('results/',tmp_FileName)),'mdl_tbl');
end

%% print average variance decomposition per variable
VD_macro_result = [mean(VD_mean(1,1:16)) mean(VD_mean(2,1:16)) mean(VD_mean(3,1:16))];
head={' ' 'Global', 'Sector' ,'Idio'};
disp('VD for macro series');
VD_m_tbl=[head; 'Macro' num2cell(VD_macro_result)]
VD_fin_result = [mean(VD_mean(1,17:end)) mean(VD_mean(2,17:end)) mean(VD_mean(3,17:end))];
head={' ' 'Global', 'Sector' ,'Idio'};
disp('VD for financial series');
VD_f_tbl=[head; 'FIN' num2cell(VD_fin_result)]
VD_tbl = [VD_m_tbl ; VD_f_tbl];
tmp_VDFileName = append('VDresults','Model_',model,'_varopt_',financial_variables(variable_option),'_p0',num2str(p0),'.mat');
save(string(fullfile('results/',tmp_VDFileName)),'VD_tbl');

clear 'tmp_FactProbFileName', 'tmp_VDFileName';

disp('... done!')