%% Reproduces Figure 2 and Figure 3
% Figure 2: Posterior distributions of the global factor standard deviations.
% Figure 3: Comparison of the different estimates of GMFCy and the unconditional GBCy.
% “Which Global Cycle? A Stochastic Factor Selection Approach for Global Macro-Financial Cycles”
% by Berger and Hienzsch (2024)

% script runs baseline model that only includes G and F (a results from the SFS)
% for model with crediot growth and model with capital inflows

financial_variables = {'Capital inflows','Capital outflows','Hot capital inflows', 'Hot capital outflows', 'Credit', 'Loans'};
financial_var = [5 1]; % select models with credit growth and model with capital inflows
Results.fin_var = [];

model_est = [1 1];  % when estimating: 0 for excluding factor, 1 for including, 2 for sampling indicator
                    % first input is for global, second for variable-specific groups and third for country


%% Other settings
rescale_option = 1; % 0: no, 1: AL, 2: trend GDP 3: standardize(unit variance) 

% Gibbs settings
b_in1  = 1500;            % burn-in where indicators are fixed
b_in2  = 1500;            % burn-in where indicators are sampled
b_in   = b_in1 + b_in2;   % full burn-in
ndraws = 30000;           % number of useful draws
it_print = 200;
thin   = 1;             % thin out, use every nth draw
lags   = 1;

% Choice normalization on loading global factor
ortho_est = 1; % 
norm = 0;      % choice for normalization on loading of global factor,
               % 0: sets mean to 1

%% Run model for capital inflows and credit growth

for fin_indx = 1:length(financial_var)

    variable_option = financial_var(fin_indx);
    disp('************************')
    disp(string(append('RUN MODEL with rGDP and ',financial_variables(variable_option))));
     disp('************************')
    %% Data 
    disp('Preparing data for analysis...');
    
    % call data import and transfom function
    y = gmfcy_data_OECD20(variable_option, rescale_option);
    
    % Assumed number of factors in each layer
    [T,I] = size(y); % time sample and total no. of data series
    J = 2;      % no. of variables per country
    N = I/J;    % no. of countries
    Gnr = 1;    % no. of global common factors (i.e. global macro-fin fact G)
    Fnr = 2;    % no. of global variable-specific factors (i.e. F^m F^f)
    M = Gnr + Fnr; % no. of common factors
    K = J/Fnr; % no. of sector-specific variables per country
    
    % Indicators for country and sector
    % macro (1) and financial (2) variable indicator k= {1,2}
    ind_f = [ones(1,I/Fnr) 2*ones(1,I/Fnr)];
    
    disp('... done');
    
    %% Binary indicators
    s_delta = ones(1,M);
    if model_est(1,1) == 0; s_delta(1,1) = 0; end
    if model_est(1,2) == 0; s_delta(2:1+Fnr) = zeros(1,Fnr); end 
    s_delta(1,2) = 0;

    % Binary indicator settings for common factors
    model_exp              = zeros(1,1+Fnr);
    model_exp(1,1)         = model_est(1,1); % global
    model_exp(1,2)         = 0; % macro
    model_exp(1,3)         = model_est(1,2); % fin
    
    %% Priors (hyperparameters)
    % Normal
    % standard deviations of factors
    a0_std_fac = 0;                 A0_std_fac =  10;
    % factor loadings
    a0_lambda = 1;                  A0_lambda = 0.15^2;
    % AR parameters
    a0_ar = ones(lags,1)*0.5/lags;  A0_ar     = eye(lags)*0.15^2; 
    
    % Inverse Gamma
    % error variance of nu
    bel_var_mu = 1;     % belief 
    str_var_mu = 0.1;   % strength of belief   
    c0 = T*str_var_mu;              C0 = c0*bel_var_mu;
    
    % Prior inclusion probbility (Bernoulli)
    p0 = 0.5;
    
    %% Initialize 
    % States
    s_G = randn(T,Gnr)*0;   s_GG = [];
    s_F = randn(T,Fnr)*0;   s_FF = [];
    % Stdv states
    s_sig_g = ones(Gnr,1); s_sig_gg = [];
    s_sig_f = ones(Fnr,1); s_sig_ff = [];
    s_sig_mu = ones(I,1);
    % Loadings 
    s_gam_g = ones(I,1);
    s_gam_f = ones(I,1);
    % AR Parameters
    s_phi_g =  ones(Gnr,lags)*0.5./lags;
    s_phi_f =  ones(Fnr,lags)*0.5./lags;
    s_phi_mu = ones(I,lags)*0.5./lags;
    
    %% Storage arrays
    % store states
    store_G = NaN(T,1,ndraws); store_F = NaN(T,Fnr,ndraws);
    % store loadings
    store_gam_g = NaN(I,ndraws); store_gam_f = NaN(I,ndraws);
    % store stdv 
    store_sig_g = NaN(Gnr,ndraws); store_sig_f = NaN(Fnr,ndraws); store_sig_mu = NaN(I,ndraws);
    % store phi (AR)
    store_phi_g = NaN(Gnr,ndraws); store_phi_f = NaN(Fnr,ndraws); store_phi_i = NaN(I,ndraws);
    % store delta
    store_delta = NaN(M,ndraws); store_p_delta = zeros(M,ndraws); 
    % store VD
    store_VD = zeros(I,3,ndraws);
    % non-real numbers counter
    count_nonreal = 0;

    % Gibbs sampling
    tic;
    disp('Starting up the Gibbs sampler...');
    disp('Number of iterations');
    timing_start = now;
    h = waitbar(0,'simulation running');
    for j=1:b_in+ndraws
        if mod(j,it_print) == 0
            disp(j);
        end    
        % ---------------------------------------------------------------------
        % Sequential sampling of the common factors from the conditional distribution $p(f|y,\beta)$
        % ---------------------------------------------------------------------

        factor_order = [2,1];
        for ii = 1:2
            switch factor_order(ii)
                case 1 
                % Variable-specific factors (f^m f^f)
                [s_FF, s_sig_ff] = states_F(y,...
                                            s_G,...
                                            s_phi_f, s_phi_mu,...
                                            s_sig_g, s_sig_f, s_sig_mu,...
                                            s_gam_g, s_gam_f,...
                                            ind_f, lags, ortho_est);
                % ************************
                real_states = isreal(s_FF);
                if real_states == 1
                s_F = s_FF;
                s_sig_f = s_sig_ff;
                elseif real_states == 0
                count_nonreal = count_nonreal + 1;
                end
                % ************************   
                case 2 
                % Global factor (g^mf)
                [s_GG, s_sig_gg] = states_G(y,...
                                            s_F,...
                                            s_phi_g, s_phi_mu,...
                                            s_sig_g, s_sig_f, s_sig_mu,...
                                            s_gam_g, s_gam_f,...
                                            ind_f, lags, ortho_est);                  
                % ************************
                real_states = isreal(s_GG);
                if real_states == 1
                    s_G = s_GG;
                    s_sig_g = s_sig_gg;
                elseif real_states == 0
                    count_nonreal = count_nonreal + 1;
                end 
                % ************************
            end
        end
          
        % ---------------------------------------------------------------------
        % Sample the parameters \beta from the conditional distribution p(\beta|y,f)
        % ---------------------------------------------------------------------  
        
        % Stdv factors and binary indicators
        if j > b_in1 % burn-in where indicators are fixed
            s_indic = 1; 
        else
            s_indic = 0;
        end
        [s_sig_g,s_sig_f,...
            s_delta, s_p_delta] = para_stdv(y,s_G,s_F,...
                                           s_phi_mu,s_sig_mu,...
                                           a0_std_fac,A0_std_fac,...
                                           s_gam_g,s_gam_f,...
                                           ind_f,s_indic,...
                                           p0,s_delta,...
                                           model_exp);    
        
        % Loadings and var_mu
        [s_gam_g,s_gam_f, ...
            s_sig_mu,s_sig_g,s_sig_f] = para_loadings(y,s_G,s_F,...
                                                    s_phi_mu,...
                                                    s_sig_g,s_sig_f,...
                                                    a0_lambda,A0_lambda,c0,C0,...
                                                    ind_f,...
                                                    model_est,norm);    
        
        % AR parameters of factors and idiosyncratic component
        [s_phi_g,s_phi_f,s_phi_mu] = para_ar(y,s_G,s_F,...
                                                    a0_ar,A0_ar,...
                                                    s_gam_g,s_gam_f,...
                                                    s_sig_g,s_sig_f,s_sig_mu,...
                                                    s_phi_g,s_phi_f,s_phi_mu,...
                                                    ind_f,...
                                                    model_est,lags);    
        
        
        % ---------------------------------------------------------------------
        % Variance decomposition
        % ---------------------------------------------------------------------
    
        VD = zeros(I,3);
        if lags == 1
            for i = 1:I
                total_var = s_gam_g(i,1)^2*s_sig_g^2/(1-s_phi_g^2)+...
                                s_gam_f(i,1)^2*s_sig_f(ind_f(1,i),1)^2/(1-s_phi_f(ind_f(1,i),1)^2)+...
                                        s_sig_mu(i,1)/(1-s_phi_mu(i,1))^2;
                VD(i,1)   = s_gam_g(i,1)^2*s_sig_g^2/(1-s_phi_g^2)/total_var;
                VD(i,2)   = s_gam_f(i,1)^2*s_sig_f(ind_f(1,i),1)^2/(1-s_phi_f(ind_f(1,i),1)^2)/total_var;
                VD(i,3)   = s_sig_mu(i,1)/(1-s_phi_mu(i,1))^2/total_var;
                
            end
        end
        
        % ---------------------------------------------------------------------
        % Store results
        % ---------------------------------------------------------------------
        if (j>b_in) && (mod(j,thin)==0)
            % loadings
            store_gam_g(:,j-b_in) = s_gam_g; 
            store_gam_f(:,j-b_in) = s_gam_f;
            % sign switch for stdv
            sign=1; if rand>0.5; sign =-1; end
            store_sig_g(:,j-b_in) = sign*s_sig_g;
            store_sig_f(:,j-b_in) = sign*s_sig_f;
            % store error variance
            store_sig_mu(:,j-b_in)= s_sig_mu.^2;
            % AR parameters
    %         store_phi_i(:,j-b_in) = s_phi_mu;
    %         store_phi_g(:,j-b_in) = s_phi_g;
    %         store_phi_f(:,j-b_in) = s_phi_f;
            
            if size(s_G,1)>0
                store_G(:,j-b_in) = s_G*s_sig_g;
            end
            for f = 1:Fnr
                store_F(:,f,j-b_in) = s_F(:,f)*s_sig_f(f,:);
            end
    
            store_delta(:,j-b_in)   = s_delta(1,:)';
            store_p_delta(:,j-b_in) = s_p_delta(1,:)';
            
            store_VD(:,:,j-b_in)  = VD;
        end    
        
        % progress report
        if ~rem(j,10)
            waitbar(j/(b_in+ndraws), h, timing_message(j, b_in+ndraws, timing_start))
        end
    end
    toc;
    close(h)
    disp('... done');
    
    % Summarizing the results
    disp('Summarize and store results...');
    % summary of results
    results.G  = summary(store_G);
    results.F  = summary(store_F);
     
    results.sig_g  = summary((store_sig_g));
    results.sig_f  = summary((store_sig_f));
    
    results.VD  = summary(store_VD);
    
    VD_mean = results.VD.mean';
    
    results.gam_g  = summary(store_gam_g);
    results.gam_f  = summary(store_gam_f);
    results.pdelta = mean(store_p_delta,2);
    results.delta  = mean(store_delta,2);

    % results for plotting Figure 1
    Results.fin_var(fin_indx).G = results.G;
    Results.fin_var(fin_indx).F = results.F;
    disp('... done');
end

%% Plot Figure 2
credit = [];
credit.results.G.mean = Results.fin_var(1).G.mean;
credit.results.G.perc5 = Results.fin_var(1).G.perc5;
credit.results.G.perc95 = Results.fin_var(1).G.perc95;
credit.results.F.mean = Results.fin_var(1).F.mean;
credit.results.F.perc5 = Results.fin_var(1).F.perc5;
credit.results.F.perc95 = Results.fin_var(1).F.perc95;

inflows = [];
inflows.results.G.mean = Results.fin_var(2).G.mean;
inflows.results.G.perc5 = Results.fin_var(2).G.perc5;
inflows.results.G.perc95 = Results.fin_var(2).G.perc95;
inflows.results.F.mean = Results.fin_var(2).F.mean;
inflows.results.F.perc5 = Results.fin_var(2).F.perc5;
inflows.results.F.perc95 = Results.fin_var(2).F.perc95;


x = 1996.5:.25:2020.00;

figure(2);
%GDP & credit
subplot(2,2,1) % GMFCy
q = plot(x,credit.results.G.mean,x,credit.results.G.perc5,x,credit.results.G.perc95); % To get bounds y-axis 
recession_bars;
delete(q);
xconf = [x x(end:-1:1)];       
y1conf = [credit.results.G.perc5' credit.results.G.perc95(end:-1:1)'];
hold on
p1 = fill(xconf,y1conf,'blue');
p1.FaceColor = [0 0.4470 0.7410];
p1.FaceAlpha = 0.25;
p1.EdgeColor = 'none';           
plot(x,credit.results.G.mean,'Color',[0.0, 0.0, 0.5],'LineWidth',1.5)
title('Global Macro-Fin. Cycle','interpreter','latex')

subplot(2,2,2) % GFCy
q = plot(x,credit.results.F.mean(:,2),x,credit.results.F.perc5(:,2),x,credit.results.F.perc95(:,2)); % To get bounds y-axis 
recession_bars;
delete(q);
% Outflows
xconf = [x x(end:-1:1)];       
y1conf = [credit.results.F.perc5(:,2)' credit.results.F.perc95(end:-1:1,2)'];
hold on
p1 = fill(xconf,y1conf,'red');
p1.FaceColor = [0.6350 0.0780 0.1840];
p1.FaceAlpha = 0.25;
p1.EdgeColor = 'none';           
plot(x,credit.results.F.mean(:,2),'Color',[0.6350 0.0780 0.1840],'LineWidth',1.5)
title('Global Credit Cycle','interpreter','latex')

%GDP & Gross Capital Inflows
subplot(2,2,3) % GMFCy
q = plot(x,inflows.results.G.mean,x,inflows.results.G.perc5,x,inflows.results.G.perc95); % To get bounds y-axis 
recession_bars;
delete(q);
% Outflows
xconf = [x x(end:-1:1)];       
y1conf = [inflows.results.G.perc5' inflows.results.G.perc95(end:-1:1)'];
hold on
p1 = fill(xconf,y1conf,'blue');
p1.FaceColor = [0 0.4470 0.7410];
p1.FaceAlpha = 0.25;
p1.EdgeColor = 'none';           
plot(x,inflows.results.G.mean,'Color',[0.0, 0.0, 0.5],'LineWidth',1.5)
title('Global Macro-Fin. Cycle','interpreter','latex')

subplot(2,2,4) % GFCy
q = plot(x,inflows.results.F.mean(:,2),x,inflows.results.F.perc5(:,2),x,inflows.results.F.perc95(:,2)); % To get bounds y-axis 
recession_bars;
delete(q);
% Outflows
xconf = [x x(end:-1:1)];       
y1conf = [inflows.results.F.perc5(:,2)' inflows.results.F.perc95(end:-1:1,2)'];
hold on
p1 = fill(xconf,y1conf,'red');
p1.FaceColor = [0.6350 0.0780 0.1840];
p1.FaceAlpha = 0.25;
p1.EdgeColor = 'none';           
plot(x,inflows.results.F.mean(:,2),'Color',[0.6350 0.0780 0.1840],'LineWidth',1.5)
title('Global Capital Flow Cycle','interpreter','latex')


if options.FigSave == 1
    print( [ pwd '\figures\Figure_2' ] , '-dpdf' );
end

%% Plot Figure 3

GBCy = uncond_GBCy;


figure(3);
% Global Macro-Financial Cycle
subplot(2,1,1)
q = plot(x,GBCy.mean); % To get bounds y-axis 
recession_bars;
delete(q);
% unconditional GBCy
plot(x,GBCy.mean,'Color',[0.0, 0.0, 0.5],'LineWidth',1)
% Credit     
plot(x,credit.results.G.mean,'--','Color',[0.4940 0.1840 0.5560],'LineWidth',1.5)
% Inflows
plot(x,inflows.results.G.mean,'Color',[0.6350 0.0780 0.1840],'LineWidth',1.5)
title('Global Macro-Financial Cycle','interpreter','latex')
%axis([x(1,1) x(1,end) -inf inf])
xlim([x(1,1) x(1,end)])
hold off

% Global Financial Cycle
subplot(2,1,2)
q = plot(x,credit.results.F.mean(:,2),x,credit.results.F.perc5(:,2),x,credit.results.F.perc95(:,2)); % To get bounds y-axis 
recession_bars;
delete(q);
% Credit        
plot(x,credit.results.F.mean(:,2),'--','Color',[0.4940 0.1840 0.5560],'LineWidth',1.5)
% Inflows
plot(x,inflows.results.F.mean(:,2),'Color',[0.6350 0.0780 0.1840],'LineWidth',1.5)

title('Global Financial Cycle','interpreter','latex')
%axis([x(1,1) x(1,end) -inf inf])
xlim([x(1,1) x(1,end)])
hold off

if options.FigSave == 1
    print( [ pwd '\figures\Figure_3' ] , '-dpdf' );
end

