function [s_gam_g,s_gam_f,...
    s_sig_mu,s_sig_g,s_sig_f] = para_loadings(y,s_G,s_F,...
                                                    s_phi_mu,...
                                                    s_sig_g,s_sig_f,...
                                                    a0_lambda,A0_lambda,c0,C0,...
                                                    ind_f,...
                                                    model,norm)
%para_loadings Draws sample of factor loadings and the observation error variance
% -------------------------------------------------------------------------
% INPUTS:
% y                            - dependent variables
% s_G, s_F                     - regressors (i.e. common factors)
% s_phi_mu                     - AR coefficient of idiosyncr component
% s_sig_g,s_sig_f              - standard deviations of common factors
% a0_lambda,A0_lambda          - Normal Priors for loadings of factors (i.e. prior for beta in Bayes regr)
% c0,C0                        - IG Priors for observation error variance
% ind_f                        - indicators for variable- and country-specific series
% model                        - vector of choices which factors to include
% norm                         - choice for normalization on loading of global factor
% -------------------------------------------------------------------------
% OUPUTS:
% s_gam_g,s_gam_f              - sampled loadings of common factors
% s_sig_mu                     - sampled error std of observation equation
% s_sig_g,s_sig_f              - normalized standard deviation of common factors
%% Get dimensions
[T,I] = size(y);
Fnr   = max(ind_f);
N     = I/max(ind_f);

%% Transforming series
y_tilde = zeros(T,I);
G_tilde = zeros(T,I);
F_tilde = zeros(T,I);

for i = 1:I
   y_tilde(:,i) = trans_ar(y(:,i),s_phi_mu(i,:)); 
   G_tilde(:,i) = trans_ar(s_G,s_phi_mu(i,:));
   F_tilde(:,i) = trans_ar(s_F(:,ind_f(1,i)),s_phi_mu(i,:));
end

%% Sampling loadings and sig_mu
s_gam_g  = zeros(I,1);
s_gam_f  = zeros(I,1);
s_sig_mu = zeros(I,1);

for i = 1:I
    % setup matrix of regressors
    x_tilde_i = [s_sig_g*G_tilde(:,i) s_sig_f(ind_f(1,i))*F_tilde(:,i)];
    % draw beta and variance from NIG posterior of regression  model
    [b_draw,var_draw] = NIG_post(y_tilde(:,i),x_tilde_i,...
                                a0_lambda*ones(size(x_tilde_i,2),1),A0_lambda*eye(size(x_tilde_i,2)),...
                                c0,C0);
    if model(1,1)>0 s_gam_g(i,1) = b_draw(1,1); end
    if model(1,2)>0 s_gam_f(i,1) = b_draw(2,1); end
    
    s_sig_mu(i,1) = sqrt(var_draw);
end

%% Normalization for identification
%  i.e. average loadings on a factor are set to 1
%  loadings of non-selected factors are set to 0

% Global factor
if model(1,1)>0
    if norm==0
        means_G = mean(s_gam_g(1:I,1));
    elseif norm==1
        means_G = zeros(1,Fnr);
        for f = 1:Fnr
            means_G(1,f) = mean(s_gam_g(ind_f==f));
        end
    end
    s_sig_g = s_sig_g*means_G;
    s_gam_g = s_gam_g/means_G;
end

% Variables-specific factor
if model(1,2)>0
    for f = 1:Fnr
        s_sig_f(f,1) = s_sig_f(f,1)*mean(s_gam_f(ind_f==f));
        s_gam_f(ind_f==f) = s_gam_f(ind_f==f)/mean(s_gam_f(ind_f==f));
    end
end

end

