function [s_phi_g,s_phi_f,s_phi_mu] = para_ar(y,s_G,s_F,...
                                                a0_ar,A0_ar,...
                                                s_gam_g,s_gam_f,...
                                                s_sig_g,s_sig_f,s_sig_mu,...
                                                s_phi_g,s_phi_f,s_phi_mu,...
                                                ind_f,...
                                                model,lags)
%para_ar Generates a draw for AR-parameters of factor state equation and
% idiosyncratic component dynamics
%   Uses a Metropolis-Hasting step to ensure stationarity
% -------------------------------------------------------------------------
% INPUTS:
% y                            - data
% s_G, s_F                     - common factors
% s_phi_g,s_phi_f              - AR coefficients for common factors (last accepted draw)
% s_phi_mu                     - AR coefficient of idiosyncr component (last accepted draw)
% s_sig_g,s_sig_f              - standard deviations of common factors
% a0_ar,A0_ar                  - Normal Priors for AR-coefficients of factors (i.e. prior for beta in Bayes regr)
% ind_f                        - indicators for variable- and country-specific series
% model                        - vector of choices which factors to include
% lags                         - number of included lags in AR
% -------------------------------------------------------------------------
% OUPUTS:
% s_phi_g,s_phi_f              - sampled AR parameters of common factors (new draw)
% s_phi_mu                     - sampled AR parameters of idiosyncratic component (new draw)
%% Get dimensions
[T,I] = size(y);
Fnr   = max(ind_f);
N     = I/max(ind_f);
bound = 0.99;

%% Global factor
if model(1,1)>0
    % generate candidate draw
    var_d = 1;
    
    yy = s_G;
    for a = 1:lags
        yy = [yy lagmatrix(s_G,a)];
    end
    yy = yy(lags+1:end,:);
    s_phi_g_c = N_post(yy(:,1),yy(:,2:end),a0_ar,A0_ar,var_d);
    
    % MH step to ensure stationarity
    if sum(s_phi_g_c) < bound
        % Calculate acceptance probability
        YY = s_G(1:lags,1);
        
        V_inv1 = inv(var_ar(s_phi_g_c',1));
        V_inv2 = inv(var_ar(s_phi_g,1));
        
        prob1 = sqrt(det(V_inv1))*exp(-0.5/var_d*YY'*V_inv1*YY);
        prob2 = sqrt(det(V_inv2))*exp(-0.5/var_d*YY'*V_inv2*YY);
        
        acprob = min(prob1/prob2,1);
        if acprob > rand
            s_phi_g = s_phi_g_c';
        end
    end
end

%% Variables-specific factor
if model(1,2)>0
    for f = 1:Fnr
        % generate candidate draw
        var_d = 1;
        yy = s_F(:,f);
        for a = 1:lags
            yy = [yy lagmatrix(s_F(:,f),a)];
        end
        yy = yy(lags+1:end,:);
        s_phi_f_c = N_post(yy(:,1),yy(:,2:end),a0_ar,A0_ar,var_d);
        
    % MH step to ensure stationarity
        if sum(s_phi_f_c) < bound
            % Calculate acceptance probability
            YY = s_F(1:lags,f);
            
            V_inv1 = inv(var_ar(s_phi_f_c',1));
            V_inv2 = inv(var_ar(s_phi_f(f,:),1));
            
            prob1 = sqrt(det(V_inv1))*exp(-0.5/var_d*YY'*V_inv1*YY);
            prob2 = sqrt(det(V_inv2))*exp(-0.5/var_d*YY'*V_inv2*YY);

            acprob = min(prob1/prob2,1);
            if acprob > rand
                s_phi_f(f,:) = s_phi_f_c';
            end
        end        
    end
end

%% Idiosyncratic components
for i = 1:I
    % construct mu
    xx = 0;
    if size(s_G,1)>0 xx = xx + s_gam_g(i,1)*s_sig_g*s_G; end
    if size(s_F,1)>0 xx = xx + s_gam_f(i,1)*s_sig_f(ind_f(1,i),1)*s_F(:,ind_f(1,i)); end
    mu = y(:,i) - xx;
    var_i = s_sig_mu(i,1)^2;
   
   % generate candidate draw
   yy = mu;
   for a = 1:lags
       yy = [yy lagmatrix(mu,a)];
   end
   yy = yy(lags+1:end,:);
   s_phi_i_c = N_post(yy(:,1),yy(:,2:end),a0_ar,A0_ar,var_i);
   
    % MH step to ensure stationarity
    if sum(s_phi_i_c) < bound
        % Calculate acceptance probability
        YY = mu(1:lags,1);

        V_inv1 = inv(var_ar(s_phi_i_c',1));
        V_inv2 = inv(var_ar(s_phi_mu(i,:),1));

        prob1 = sqrt(det(V_inv1))*exp(-0.5/var_i*YY'*V_inv1*YY);
        prob2 = sqrt(det(V_inv2))*exp(-0.5/var_i*YY'*V_inv2*YY);

        acprob = min(prob1/prob2,1);
        if acprob > rand
            s_phi_mu(i,:) = s_phi_i_c;
        end
    end   
end

end

