function [Log_ML,aT,AT,cT,CT] = Bayesian_regression(y,x,...
                                                   a0,A0,c0,C0,Sigma)
%Bayesian_regression Estimates posterior parameters and calculates
%posterior marginal log likelihood of regression model
% -------------------------------------------------------------------------
% INPUTS:
% y                            - dependent variables
% x                            - regressors (i.e. common factors)
% a0, A0                       - Priors for std of factors (i.e. prior for beta in Bayes regr)
% s_gam_g, s_gam_f, s_gam_c    - loadings of common factors
% ind_f, ind_c                 - indicators for variable- and country-specific series
% 
% -------------------------------------------------------------------------
% OUPUTS:
% Log_ML                       - posterior marginal log likelihood of regression model
% aT, AT                       - posterior mean and variance for coeff
% cT, CT                       - 
% -------------------------------------------------------------------------
% Get dimensions
[T,k] = size(y);

% Estimate posterior parameters and
% calculate posterior marginal log likelihood
if Sigma == 0
    % Posterior parameters
    AT = inv(x'*x + inv(A0));
    aT = AT*(x'*y + inv(A0)*a0);
    cT = c0 + T/2;
    CT = C0 + 0.5*(y'*y+a0'*inv(A0)*a0-aT'*inv(AT)*aT);

    % Marginal likelihood of the regression model
    Log_ML = 0.5*log(det(AT))-0.5*log(det(A0)) + gammaln(cT) + c0*log(C0) - gammaln(c0) - cT*log(CT);
    
elseif Sigma == 1
    % Posterior parameters
    AT = inv(x'*x + inv(A0));
    aT = AT*(x'*y + inv(A0)*a0);
    cT = []; CT = [];
   
    % Marginal likelihood of the regression model
    Log_ML =  0.5*log(det(AT)) -0.5*log(det(A0)) -0.5*(y'*y + a0'*inv(A0)*a0 - aT'*inv(AT)*aT);
    
else
    % make diagonal matrix of vector
    Sigma = diag(Sigma); 
    Sigma_inv = inv(Sigma);
    % Posterior parameters
    AT = inv(x'*Sigma_inv*x + inv(A0));
    aT = AT*(x'*Sigma_inv*y + inv(A0)*a0);
    cT = []; CT = [];
   
    % Marginal likelihood of the regression model
    Log_ML =  0.5*log(det(AT)) -0.5*log(det(A0)) -0.5*(y'*Sigma_inv*y + a0'*inv(A0)*a0 - aT'*inv(AT)*aT);
    % -0.5*log(det(Sigma)) term is excluded because
        %(i)  it is constant over restricted and unrestricted model
        %(ii) it is INF if det(Sigma)=0, causing NaN's in the remainder of the code
    
end

end

