# Reviewer comment 3: Dimension vs Precision / Computing time (GPAL)
source("supplementary_analysis.R")


# Run GPAL for d = 5, 10, 20, 40 and track the MSE and time
library(gptools2)
set.seed(1234)


dimension <- c(5, 10, 20, 40)
seed_vector <- sample(1e9, length(dimension))

results <- map(dimension, function(p) {
    set.seed(seed_vector[which(dimension == p)])


    # - Use active learning with GP
    n <- 100
    b0 <- runif(p)
    B0 <- diag(runif(p, 0.5, 1.5))
    a0 <- 30
    d0 <- 3
    data <- sim_bayesian_linear_regression(n, p, b0, B0, a0, d0)
    X <- data$X
    y <- data$y
    n_eval <- 100
    initial_X <- matrix(rnorm(n_eval * p), n_eval, p)


    # - Fit model to data
    tictoc::tic()
    model_gps <- 1:p |>
        map(function(j) {
            f <- \(b0) mcmc_sliced_mean_with_sd(gibbs_sampler(b0, B0, a0, d0, X, y)$beta[,j])
            model <- active_learning(initial_X, f,
                                     max_iter = 100, sample_n = 100,
                                     tol = c(0.1, 0.01), persistence = 10, consecutive = 3)
        })
    time_model_fitting <- tictoc::toc()


    # - Generate testing datasets
    testing_X <- matrix(rnorm(n_eval * p), n_eval, p)

    testing_y_truth <- testing_X |>
        map_row(\(b0) t(posterior_mean_analytical(b0, B0, X, y))) |>
        reduce(rbind)

    testing_y_gp <- model_gps |>
        map(\(model) predict_gp(model$model, testing_X)$mean) |>
        reduce(cbind)


    # - Compare with the analytical formula
    error <- mse(testing_y_truth, testing_y_gp)

    # - Return results
    list(testing = list(X = testing_X, y_truth = testing_y_truth, y_gp = testing_y_gp),
         mse = error, time = time_model_fitting)
})

saveRDS(results, "output/GPAL_vs_dimension.RDS")
