library(targets)


# Set up paths
project_path <- list(
    dataset = "../dataset",
    output = "../output",
    R_files = "LAIDS"
)
dataset <- \(path) file.path(project_path$dataset, path)
output <-  \(path) file.path(project_path$output, path)


# Source R files
sapply(list.files(project_path$R_files, full.names = T), source)


# Set up project wide options
lapply2 <- set_lapply(TRUE, mc.cores = 3)


# Target plan
tar_option_set(packages = c("purrr", "systemfit", "magrittr", "gptools2",
                            "matrixcalc", "ADtools", "abind", "latex2exp"))
list(
    # 0. Load data -------------------------------------------------------------
    tar_target(data_file, dataset("FirstStage.csv"), format = "file"),
    tar_target(data0, read.table(data_file, header = T, sep=",")),

    # 1. Frequentist estimation ------------------------------------------------
    tar_target(freq_est, frequentist_estimate(data0)),
    tar_target(freq_exp_elast, freq_EE(freq_est, data0)),
    tar_target(freq_cross_elast, freq_CE(freq_est, data0)),
    tar_target(table_freq_est, list(
        writeToFile(freq_est,          output("tables/freq_est_Coefficients.txt")),
        writeToFile(summary(freq_est), output("tables/freq_est_Summary.txt")),
        writeToFile(freq_exp_elast,    output("tables/freq_est_Expenditure_Elasticity.txt")),
        writeToFile(freq_cross_elast,  output("tables/freq_est_CrossPrice_Elasticity.txt"))
    )),

    # 2. Bayesian estimation ----------------------------------------------------
    # Set up prior hyperparameters. The frequentist estimate is needed because
    # prior variance is not provided by the experts.
    tar_target(priors, setup_hyperparameter(freq_est, ratio = 1)),

    # Compute the Price Elasticity corresponding to the prior hyperparameters
    tar_target(priors_exp_elast, prior_EE(priors, data0)),
    tar_target(priors_cross_elast, prior_CE(priors, data0)),
    tar_target(table_priors, list(
        writeToFile(priors,             output("tables/priors_Coefficients.txt")),
        writeToFile(priors_exp_elast,   output("tables/priors_Expenditure_Elasticity.txt")),
        writeToFile(priors_cross_elast, output("tables/priors_CrossPrice_Elasticity.txt"))
    )),

    # Inference using Gibbs samples and computation of Price Elasticities
    tar_target(num_iter, 1e3 + 1e2),
    tar_target(bayes_samples, Gibbs_sampler(
        data = data0,
        d0 = priors$PriorMean, D0 = priors$PriorCov,
        Wv = 4, WEsc0 = diag(4), Sig = diag(4),
        num_iter = num_iter
    )),
    tar_target(bayes_EE_samples, bayes_EE(bayes_samples, data0)), # Expenditure
    tar_target(bayes_CE_samples, bayes_CE(bayes_samples, data0)), # Cross-price

    # Compute the posterior statistics
    tar_target(burn_in, 100),
    tar_target(bayes_est, list(
        mean = tidy_bayes_est(bayes_samples, burn_in, mean),
        sd = tidy_bayes_est(bayes_samples, burn_in, sd)
    )),
    tar_target(bayes_exp_elast, list(
        mean = tidy_bayes_EE(bayes_EE_samples, burn_in, mean),
        sd = tidy_bayes_EE(bayes_EE_samples, burn_in, sd)
    )),
    tar_target(bayes_cross_elast, list(
        mean = tidy_bayes_CE(bayes_CE_samples, burn_in, mean),
        sd = tidy_bayes_CE(bayes_CE_samples, burn_in, sd)
    )),
    tar_target(table_bayes_est, list(
        writeToFile(bayes_est,         output("tables/bayes_est_Coefficients.txt")),
        writeToFile(bayes_exp_elast,   output("tables/bayes_est_Expenditure_Elasticity.txt")),
        writeToFile(bayes_cross_elast, output("tables/bayes_est_CrossPrice_Elasticity.txt"))
    )),

    # 2b. Bayesian estimation with zero-prior ----------------------------------
    # Set up prior hyperparameters. The frequentist estimate is needed because
    # prior variance is not provided by the experts.
    tar_target(zero_priors, zero_prior(freq_est, ratio = 1 / 9)),

    # Compute the Price Elasticity corresponding to the zero prior hyperparameters
    tar_target(zero_exp_elast, prior_EE(zero_priors, data0)),
    tar_target(zero_cross_elast, prior_CE(zero_priors, data0)),
    tar_target(table_zero, list(
        writeToFile(zero_priors,      output("tables/zero_priors_Coefficients.txt")),
        writeToFile(zero_exp_elast,   output("tables/zero_Expenditure_Elasticity.txt")),
        writeToFile(zero_cross_elast, output("tables/zero_CrossPrice_Elasticity.txt"))
    )),

    # Inference using Gibbs samples and computation of Price Elasticities
    tar_target(zero_bayes_samples, Gibbs_sampler(
        data = data0,
        d0 = zero_priors$PriorMean, D0 = zero_priors$PriorCov,
        Wv = 4, WEsc0 = diag(4), Sig = diag(4),
        num_iter = num_iter
    )),
    tar_target(zero_bayes_EE_samples, bayes_EE(zero_bayes_samples, data0)), # Expenditure
    tar_target(zero_bayes_CE_samples, bayes_CE(zero_bayes_samples, data0)), # Cross-price

    # Compute the posterior statistics
    tar_target(zero_bayes_est, list(
        mean = tidy_bayes_est(zero_bayes_samples, burn_in, mean),
        sd = tidy_bayes_est(zero_bayes_samples, burn_in, sd)
    )),
    tar_target(zero_bayes_exp_elast, list(
        mean = tidy_bayes_EE(zero_bayes_EE_samples, burn_in, mean),
        sd = tidy_bayes_EE(zero_bayes_EE_samples, burn_in, sd)
    )),
    tar_target(zero_bayes_cross_elast, list(
        mean = tidy_bayes_CE(zero_bayes_CE_samples, burn_in, mean),
        sd = tidy_bayes_CE(zero_bayes_CE_samples, burn_in, sd)
    )),
    tar_target(table_zero_bayes_est, list(
        writeToFile(zero_bayes_est,         output("tables/zero_bayes_est_Coefficients.txt")),
        writeToFile(zero_bayes_exp_elast,   output("tables/zero_bayes_est_Expenditure_Elasticity.txt")),
        writeToFile(zero_bayes_cross_elast, output("tables/zero_bayes_est_CrossPrice_Elasticity.txt"))
    )),

    # 2c. Bayesian estimation with non-informative prior -----------------------
    # Set up prior hyperparameters. The frequentist estimate is needed because
    # prior variance is not provided by the experts.
    tar_target(non_info_priors, setup_hyperparameter(freq_est, ratio = 1 / 9)),

    # Compute the Price Elasticity corresponding to the non_info prior hyperparameters
    tar_target(non_info_exp_elast, prior_EE(non_info_priors, data0)),
    tar_target(non_info_cross_elast, prior_CE(non_info_priors, data0)),
    tar_target(table_non_info, list(
        writeToFile(non_info_priors,      output("tables/non_info_priors_Coefficients.txt")),
        writeToFile(non_info_exp_elast,   output("tables/non_info_Expenditure_Elasticity.txt")),
        writeToFile(non_info_cross_elast, output("tables/non_info_CrossPrice_Elasticity.txt"))
    )),

    # Inference using Gibbs samples and computation of Price Elasticities
    tar_target(non_info_bayes_samples, Gibbs_sampler(
        data = data0,
        d0 = non_info_priors$PriorMean, D0 = non_info_priors$PriorCov,
        Wv = 4, WEsc0 = diag(4), Sig = diag(4),
        num_iter = num_iter
    )),
    tar_target(non_info_bayes_EE_samples, bayes_EE(non_info_bayes_samples, data0)), # Expenditure
    tar_target(non_info_bayes_CE_samples, bayes_CE(non_info_bayes_samples, data0)), # Cross-price

    # Compute the posterior statistics
    tar_target(non_info_bayes_est, list(
        mean = tidy_bayes_est(non_info_bayes_samples, burn_in, mean),
        sd = tidy_bayes_est(non_info_bayes_samples, burn_in, sd)
    )),
    tar_target(non_info_bayes_exp_elast, list(
        mean = tidy_bayes_EE(non_info_bayes_EE_samples, burn_in, mean),
        sd = tidy_bayes_EE(non_info_bayes_EE_samples, burn_in, sd)
    )),
    tar_target(non_info_bayes_cross_elast, list(
        mean = tidy_bayes_CE(non_info_bayes_CE_samples, burn_in, mean),
        sd = tidy_bayes_CE(non_info_bayes_CE_samples, burn_in, sd)
    )),
    tar_target(table_non_info_bayes_est, list(
        writeToFile(non_info_bayes_est,         output("tables/non_info_bayes_est_Coefficients.txt")),
        writeToFile(non_info_bayes_exp_elast,   output("tables/non_info_bayes_est_Expenditure_Elasticity.txt")),
        writeToFile(non_info_bayes_cross_elast, output("tables/non_info_bayes_est_CrossPrice_Elasticity.txt"))
    )),

    # Baseline plots -----------------------------------------------------------
    tar_target(ee_baseline_plot,
               plot_ee_baseline_estimates(bayes_exp_elast,
                                          freq_exp_elast,
                                          non_info_bayes_exp_elast)),
    tar_target(ce_baseline_plot,
               plot_ce_baseline_estimates(bayes_cross_elast,
                                          freq_cross_elast,
                                          non_info_bayes_cross_elast)),
    # 3.1 Local analysis -----------------------------------------------------------
    # a. Get Jacobian of price elasticities with respect to the expert prior parameters
    tar_target(bayes_est_samples_AD,
               AD_Gibbs(data = data0,
                        d0 = priors$PriorMean,
                        log_D0 = log(diag(priors$PriorCov)),
                        Wv = 4, WEsc0 = diag(4), Sig = diag(4),
                        num_iter = num_iter, method = "inv_tf")),
    tar_target(bayes_est_CE_AD,
               AD_bayes_CE(bayes_est_samples_AD, data0)),
    tar_target(bayes_est_EE_AD,
               AD_bayes_EE(bayes_est_samples_AD, data0)),
    # (coefficients, cross-price elasticities and expenditure elasticities)
    tar_target(bayes_est_AD_Jacobian,
               get_Jacobian(tidy_AD(bayes_est_samples_AD))),
    tar_target(bayes_est_CE_AD_Jacobian,
               get_Jacobian_CE(tidy_AD_bayes_CE(bayes_est_CE_AD))),
    tar_target(bayes_est_EE_AD_Jacobian,
               get_Jacobian_EE(tidy_AD_bayes_EE(bayes_est_EE_AD))),

    # b. Get Jacobian of price elasticities with respect to the zero prior parameters
    tar_target(zero_bayes_est_samples_AD,
               AD_Gibbs(data = data0,
                        d0 = zero_priors$PriorMean,
                        log_D0 = log(diag(zero_priors$PriorCov)),
                        Wv = 4, WEsc0 = diag(4), Sig = diag(4),
                        num_iter = num_iter, method = "inv_tf")),
    tar_target(zero_bayes_est_CE_AD,
               AD_bayes_CE(zero_bayes_est_samples_AD, data0)),
    tar_target(zero_bayes_est_EE_AD,
               AD_bayes_EE(zero_bayes_est_samples_AD, data0)),
    # (coefficients, cross-price elasticities and expenditure elasticities)
    tar_target(zero_bayes_est_AD_Jacobian,
               get_Jacobian(tidy_AD(zero_bayes_est_samples_AD))),
    tar_target(zero_bayes_est_CE_AD_Jacobian,
               get_Jacobian_CE(tidy_AD_bayes_CE(zero_bayes_est_CE_AD))),
    tar_target(zero_bayes_est_EE_AD_Jacobian,
               get_Jacobian_EE(tidy_AD_bayes_EE(zero_bayes_est_EE_AD))),


    # c. Get Jacobian of price elasticities with respect to the non-informative prior parameters
    tar_target(non_info_bayes_est_samples_AD,
               AD_Gibbs(data = data0,
                        d0 = non_info_priors$PriorMean,
                        log_D0 = log(diag(non_info_priors$PriorCov)),
                        Wv = 4, WEsc0 = diag(4), Sig = diag(4),
                        num_iter = num_iter, method = "inv_tf")),
    tar_target(non_info_bayes_est_CE_AD,
               AD_bayes_CE(non_info_bayes_est_samples_AD, data0)),
    tar_target(non_info_bayes_est_EE_AD,
               AD_bayes_EE(non_info_bayes_est_samples_AD, data0)),
    # (coefficients, cross-price elasticities and expenditure elasticities)
    tar_target(non_info_bayes_est_AD_Jacobian,
               get_Jacobian(tidy_AD(non_info_bayes_est_samples_AD))),
    tar_target(non_info_bayes_est_CE_AD_Jacobian,
               get_Jacobian_CE(tidy_AD_bayes_CE(non_info_bayes_est_CE_AD))),
    tar_target(non_info_bayes_est_EE_AD_Jacobian,
               get_Jacobian_EE(tidy_AD_bayes_EE(non_info_bayes_est_EE_AD))),

    # Plot the Jacobian matrices (a, b, c)
    tar_target(jacobian_expert,
               plot_jacobian(bayes_est_CE_AD_Jacobian,
                             bayes_est_EE_AD_Jacobian,
                             color_symmetry_cap = TRUE,
                             suffix = "expert",
                             save_to_file = TRUE)),
    tar_target(jacobian_zero,
               plot_jacobian(zero_bayes_est_CE_AD_Jacobian,
                             zero_bayes_est_EE_AD_Jacobian,
                             color_symmetry_cap = TRUE,
                             suffix = "zero",
                             save_to_file = TRUE,
                             scale_of_variance = 10^-2)),
    tar_target(jacobian_non_info,
               plot_jacobian(non_info_bayes_est_CE_AD_Jacobian,
                             non_info_bayes_est_EE_AD_Jacobian,
                             color_symmetry_cap = TRUE,
                             suffix = "non_info",
                             save_to_file = TRUE,
                             scale_of_variance = 10^-2)),

    # Get the most influential input parameters and the most sensitive output parameters
    tar_target(influential_input_local,
               which_influential_k(jacobian_expert, k = 3)),
    tar_target(influential_input_local_location,
               which_influential_k(jacobian_expert, k = 5, subset = 1:18)),
    tar_target(influential_input_local_precision,
               which_influential_k(jacobian_expert, k = 5, subset = 19:36)),

    tar_target(sensitive_output_local,
               which_sensitive_k(jacobian_expert, k = 3)),
    tar_target(sensitive_output_local_location,
               which_sensitive_k(jacobian_expert, k = 5, subset = 1:18)),
    tar_target(sensitive_output_local_precision,
               which_sensitive_k(jacobian_expert, k = 5, subset = 19:36)),

    # ========================== Posterior manifolds ===========================
    # 3.2. Scenario spectrum with confidence ratio -----------------------------
    # - Ground truth
    tar_target(confidence_weights, seq(0.1, 0.9, 0.05)),
    tar_target(confidence_ratios, convert_weights_to_ratios(confidence_weights)),

    # - Run Gibbs sampler on varying confidence of the prior and compute all the
    # price elasticities
    tar_target(bayes_ratio_batch, confidence_ratios %>%
                   map(function(ratio) setup_hyperparameter(freq_est, ratio = ratio)) %>%
                   lapply2(Gibbs_sampler_pipe,
                           data = data0, num_iter = 1e4 + 1e3, burn_in = 1e3,
                           Wv = 4, WEsc0 = diag(4), Sig = diag(4))),

    tar_target(bayes_EE_batch, bayes_ratio_batch %>% map(~.x$bayes_exp_elast)),
    tar_target(bayes_CE_batch, bayes_ratio_batch %>% map(~.x$bayes_cross_elast)),

    # - Estimate posterior curve based on the evaluated posterior statistics
    tar_target(bayes_EE_GP, estimate_EE_manifolds_on_weights(confidence_weights, bayes_EE_batch)),
    tar_target(bayes_CE_GP, estimate_CE_manifolds_on_weights(confidence_weights, bayes_CE_batch)),

    # - Plot the manifold on scenario spectrum
    tar_target(bayes_spectrum_plot,
               plot_spectrum(bayes_CE_GP, bayes_EE_GP,
                             freq_cross_elast, freq_exp_elast,
                             priors_cross_elast, priors_exp_elast,
                             suffix = "expert_confidence")),

    # 3.2 Scenario spectrum with zero prior and expert prior -------------------
    tar_target(convex_weights, seq(0.1, 0.9, 0.05)),
    tar_target(zero_expert_batch,
               convex_weights %>%
                   map(function(w) linear_interpolate(w, zero_priors, priors)) %>%
                   lapply2(Gibbs_sampler_pipe,
                           data = data0, num_iter = 1e4 + 1e3, burn_in = 1e3,
                           Wv = 4, WEsc0 = diag(4), Sig = diag(4))),

    tar_target(zero_expert_EE_batch, zero_expert_batch %>% map(~.x$bayes_exp_elast)),
    tar_target(zero_expert_CE_batch, zero_expert_batch %>% map(~.x$bayes_cross_elast)),

    # - Estimate posterior curve based on the evaluated posterior statistics
    tar_target(zero_expert_EE_GP, estimate_EE_manifolds_on_weights(convex_weights, zero_expert_EE_batch)),
    tar_target(zero_expert_CE_GP, estimate_CE_manifolds_on_weights(convex_weights, zero_expert_CE_batch)),

    # - Plot the manifold on scenario spectrum
    tar_target(zero_expert_spectrum_plot,
               plot_spectrum(zero_expert_CE_GP, zero_expert_EE_GP,
                             zero_cross_elast, zero_exp_elast,
                             priors_cross_elast, priors_exp_elast,
                             suffix = "expert_vs_zero",
                             xlab = "Weight on expert prior information",
                             main_prior = "Expert prior",
                             alter_prior = "Zero prior",
                             title_ylab_line = 5,
                             mar = c(6.1, 8.1, 4.1, 2.1),
                             legend_x = c(0.09, 0.12))),
    # 3.3 High sensitivity region ----------------------------------------------
    tar_target(AD_prior_input, c(priors$PriorMean, log(diag(priors$PriorCov)))),
    tar_target(gradient_trajectory,
               bidirectional_gradient_climb(AD_prior_input, data0, num_iter, num_steps = 5)),

    # Result 1: Get the domain, range and aggregate Jacobian along the trajectory
    tar_target(trajectory_domain,
               get_trajectory_domain(gradient_trajectory)),
    tar_target(trajectory_range,
               get_trajectory_range(gradient_trajectory)),
    tar_target(trajectory_jacobian,
               get_trajectory_jacobian(gradient_trajectory)),
    # Measures
    tar_target(trajectory_domain_bound,
               t(apply(trajectory_domain, 1, range))),
    tar_target(trajectory_range_bound,
               t(apply(trajectory_range, 1, range))),
    tar_target(aggregate_jacobian,
               reduce(trajectory_jacobian, `+`) / length(trajectory_jacobian)),
    tar_target(trajectory_jacobian_norm,
               sapply(trajectory_jacobian, \(x) max(abs(x)))[c(11:7, 1:6)]),
    # Generate aggregate Jacobian plot
    tar_target(aggregate_jacobian_plot,
               plot_aggregate_jacobian(aggregate_jacobian,
                                       color_symmetry_cap = TRUE,
                                       suffix = "expert",
                                       save_to_file = TRUE)),
    tar_target(aggregate_jacobian_plot_comparions,
               plot_aggregate_jacobian(aggregate_jacobian |>
                                           # for comparison with the local analysis
                                           clip_columns(1.51, 1:18) |>
                                           clip_columns(0.13, 19:36),
                                       color_symmetry_cap = TRUE,
                                       suffix = "expert_clipped_for_comparison",
                                       save_to_file = TRUE)),
    # Get the most influential input parameters and the most sensitive output parameters
    tar_target(influential_input_aggregate,
               which_influential_k(aggregate_jacobian_plot, k = 5)),
    tar_target(sensitive_output_aggregate,
               which_sensitive_k(aggregate_jacobian_plot, k = 5)),
    # Get high sensitivity region
    tar_target(high_sensitivity_region,
               get_high_sensitivity_region(trajectory_domain_bound,
                                           influential_input_aggregate)),
    tar_target(initial_grid,
               generate_grid_from_bound(high_sensitivity_region, k = 2)),
    # Generate aggregate domain (location)
    tar_target(influential_input_aggregate_location,
               which_influential_k(aggregate_jacobian_plot, k = 5, subset = 1:18)),
    tar_target(aggregate_domain_table_location,
               domain_table(priors,
                            trajectory_domain_bound,
                            influential_input_aggregate_location)),
    tar_target(high_sensitivity_region_location,
               get_high_sensitivity_region(trajectory_domain_bound,
                                           influential_input_aggregate_location)),
    # Generate aggregate domain (precision)
    tar_target(influential_input_aggregate_precision,
               which_influential_k(aggregate_jacobian_plot, k = 5, subset = 19:36)),
    tar_target(aggregate_domain_table_precision,
               domain_table(priors,
                            trajectory_domain_bound,
                            influential_input_aggregate_precision)),
    tar_target(high_sensitivity_region_precision,
               get_high_sensitivity_region(trajectory_domain_bound,
                                           influential_input_aggregate_precision)),

    # Build 5-dimensional manifold for the high sensitivity region -------------
    # key: "bayes_exp_elast" or "bayes_cross_elast"
    # We will inspect the most sensitive output one at a time manually to avoid
    # having one big cache object. Run `tar_read(sensitive_output_aggregate)`
    tar_target(max_iter, 100),
    tar_target(sample_n, 100),
    tar_target(persistence, 10),
    tar_target(consecutive, 5),
    tar_target(mc.cores, 16),
    #
    tar_target(configured_active_learning,
               partial(active_learning,
                       max_iter = max_iter,
                       sample_n = sample_n,
                       persistence = persistence,
                       consecutive = consecutive,
                       tol = c(0.1, NA),
                       parallel = TRUE,
                       mc.cores = mc.cores)),

    tar_target(EE_1_manifold, configured_active_learning(
        as.matrix(initial_grid),
        partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
                key = "bayes_exp_elast", index = 1)
    )),
    tar_target(EE_3_manifold, configured_active_learning(
        as.matrix(initial_grid),
        partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
                key = "bayes_exp_elast", index = 3)
    )),
    tar_target(CE_14_manifold, configured_active_learning(
        as.matrix(initial_grid),
        partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
                key = "bayes_cross_elast", index = 4)
    )),
    tar_target(CE_12_manifold, configured_active_learning(
        as.matrix(initial_grid),
        partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
                key = "bayes_cross_elast", index = 2)
    )),
    tar_target(CE_34_manifold, active_learning(
        as.matrix(initial_grid),
        partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
                key = "bayes_cross_elast", index = 18)
    )),
    #
    # # Variation table ----------------------------------------------------------
    tar_target(manifold_variations,
               lapply(list(EE_1_manifold,
                           EE_3_manifold,
                           CE_12_manifold,
                           CE_14_manifold,
                           CE_34_manifold),
                      manifold_variation) %>%
                   do.call(rbind, .)),

    # Visualisation (pair high sensitivity prior mean parameters with their variances)----
    # Basic mean-variance plots ----
    tar_target(vis_active_learning,
               partial(active_learning,
                       max_iter = max_iter,
                       sample_n = sample_n,
                       persistence = persistence,
                       consecutive = consecutive,
                       tol = c(0.1, NA),
                       parallel = TRUE,
                       mc.cores = mc.cores)),
    tar_target(vis_bound_mv_list, mean_to_pairs(high_sensitivity_region_location,
                                                trajectory_domain_bound)),
    tar_target(vis_grid,
               lapply(vis_bound_mv_list, generate_grid_from_bound, k = 4)),
    tar_target(vis_grid_extended,
               lapply(vis_bound_mv_list, generate_grid_from_bound, k = 10)),
    # tar_target(vis_EE_1_manifold,
    #            vis_active_learning(
    #                as.matrix(vis_grid[[5]]),
    #                partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
    #                        key = "bayes_exp_elast", index = 1))),
    # tar_target(vis_CE_14_manifold,
    #            vis_active_learning(
    #                as.matrix(vis_grid[[5]]),
    #                partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
    #                        key = "bayes_cross_elast", index = 4))),
    tar_target(vis_EE_3_manifold,
               vis_active_learning(
                   as.matrix(vis_grid[[4]]),
                   partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
                           key = "bayes_exp_elast", index = 3))),
    # tar_target(vis_CE_12_manifold,
    #            vis_active_learning(
    #                as.matrix(vis_grid[[3]]),
    #                partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
    #                        key = "bayes_cross_elast", index = 2))),
    # tar_target(vis_CE_34_manifold,
    #            vis_active_learning(
    #                as.matrix(vis_grid[[4]]),
    #                partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
    #                        key = "bayes_cross_elast", index = 18))),

    # Explore a bigger range: mean-expanded variance plots ----
    tar_target(vis_active_learning_2,
               partial(active_learning,
                       max_iter = max_iter,
                       sample_n = sample_n,
                       persistence = persistence,
                       consecutive = consecutive,
                       tol = c(0.1, NA),
                       parallel = TRUE,
                       mc.cores = 25)),
    tar_target(vis_bound_mv_list_2,
               lapply(vis_bound_mv_list,
                      replace_variance_UB,
                      freq_est = freq_est,
                      ratio = 1 / 9)),
    tar_target(vis_grid_2,
               lapply(vis_bound_mv_list_2, generate_grid_from_bound, k = 10)),
    tar_target(vis_CE_14_manifold_2,
               vis_active_learning_2(
                   as.matrix(vis_grid_2[[3]]),
                   partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
                           key = "bayes_cross_elast", index = 4))),

    # Expanded variance-variance plots ----
    tar_target(var_bound,
               get_high_sensitivity_region(trajectory_domain_bound,
                                           influential_input_aggregate_precision)),
    tar_target(var_bound_2,
               replace_variance_UB(var_bound, freq_est, 1 / 9)),
    tar_target(var_grid,
               generate_grid_from_bound(var_bound_2[, c(3, 5)], k = 10)),

    tar_target(vis_CE_34_manifold_3,
               vis_active_learning_2(
                   as.matrix(var_grid),
                   partial(mcmc_evaluation, prior_skeleton = priors, data = data0,
                           key = "bayes_cross_elast", index = 18))),
    # Plots
    tar_target(p1,
               plot_manifold(as.matrix(vis_grid_extended[[4]]),
                             vis_EE_3_manifold,
                             xlab = "Prior mean of beta[3]",
                             ylab = "Prior (log-)variance of beta[3]",
                             zlab = "Posterior mean of eta[3]") %>%
                   layout(scene = list(aspectmode = "manual",
                                       aspectratio = list(x=1, y=1, z=1),
                                       camera = list(eye = list(x = -1.25, y = 1.25, z = 1.25))))),
    tar_target(p2,
               plot_manifold(as.matrix(vis_grid_2[[3]]),
                             vis_CE_14_manifold_2,
                             xlab = "Prior mean of gamma[1,2]",
                             ylab = "Prior (log-)variance of gamma[1,2]",
                             zlab = "Posterior mean of epsilon[1,4]") %>%
                   layout(scene = list(camera = list(eye = list(x = 1.5, y = 1.4, z = 0.4))))),
    tar_target(p3,
               plot_manifold(as.matrix(var_grid),
                             vis_CE_34_manifold_3,
                             xlab = "Prior (log-)variance of gamma[3,4]",
                             ylab = "Prior (log-)variance of gamma[4,4]",
                             zlab = "Posterior mean of epsilon[3,4]") %>%
                   layout(scene = list(camera = list(eye = list(x = 1.5, y = 1.5, z = 0.3))))),
    # Version without axis labels. The labels will be added using LaTeX,
    # since plotly does not support math rendering in 3d mode.
    tar_target(p1b, plot_manifold(as.matrix(vis_grid_extended[[4]]),
                                  vis_EE_3_manifold,
                                  xlab = "", ylab = "", zlab = "",
                                  tickfont = 18) %>%
                   layout(scene = list(aspectmode = "manual",
                                       aspectratio = list(x=1, y=1, z=1),
                                       camera = list(eye = list(x = -1.25, y = 1.25, z = 1.25))))),
    tar_target(p2b, plot_manifold(as.matrix(vis_grid_2[[3]]),
                                  vis_CE_14_manifold_2,
                                  xlab = "", ylab = "", zlab = "",
                                  tickfont = 18) %>%
                   layout(scene = list(camera = list(eye = list(x = 1.5, y = 1.4, z = 0.4))))),
    tar_target(p3b, plot_manifold(as.matrix(var_grid),
                                  vis_CE_34_manifold_3,
                                  xlab = "", ylab = "", zlab = "",
                                  tickfont = 18) %>%
                   layout(scene = list(camera = list(eye = list(x = 1.5, y = 1.5, z = 0.3)))))
)
