#' Find high sensitivity region using gradient ---------------------------------
#'
#' @param current_prior The prior hyperparameter to begin with.
#' @param data0 The data.
#' @param num_iter Number of iterations for the MCMC inference.
#' @param num_steps Number of gradient steps to take.
#'
#' @export
bidirectional_gradient_climb <- function(current_prior, data0, num_iter, num_steps = 1) {
    step <- next_step(current_prior, data0, num_iter, direction = 1)
    down <- gradient_climb(current_prior - step$pca_gradient, data0, num_iter, num_steps, direction = -1)
    up <- gradient_climb(current_prior + step$pca_gradient, data0, num_iter, num_steps, direction = 1)
    list(current = step, down_path = down, up_path = up)
}

gradient_climb <- function(current_prior, data0, AD_iter, num_steps = 1, direction = 1, ...) {
    res <- vector("list", num_steps)
    working_prior <- current_prior
    for (i in 1:num_steps) {
        working_res <- next_step(working_prior, data0, AD_iter, direction = direction, ...)
        working_prior <- working_res$next_param
        res[[i]] <- working_res
    }
    res
}

# This compute the gradient at the current input parameter and the parameter in
# the next step without stepping into it.
next_step <- function(AD_prior_input, data0, AD_iter, direction = 1, step_size) {
    AD_res <- AD_Gibbs(data = data0,
                       d0 = AD_prior_input[1:18],
                       log_D0 = AD_prior_input[19:36],
                       Wv = 4, WEsc0 = diag(4), Sig = diag(4),
                       num_iter = AD_iter, method = "inv_tf")
    # Get Jacobian for the coefficients, cross-price elasticity and expenditure elasticity
    tidy_AD_res <- tidy_AD(AD_res)
    Jacobian_coef <- get_Jacobian(tidy_AD_res)

    tidy_CE_res <- AD_res %>%
        AD_bayes_CE(data0 = data0) %>%
        tidy_AD_bayes_CE()
    Jacobian_CE <- get_Jacobian_CE(tidy_CE_res)

    tidy_EE_res <- AD_res %>%
        AD_bayes_EE(data0 = data0) %>%
        tidy_AD_bayes_EE()
    Jacobian_EE <- get_Jacobian_EE(tidy_EE_res)

    # Calculate the gradient step
    Jacobian <- rbind(Jacobian_CE, Jacobian_EE)
    pca_direction <- most_sensitive_direction(Jacobian)
    gradient <- pca_direction$gradient
    variation <- pca_direction$variation

    # Keep a record of the tidy posterior output
    AD_posterior_output <- unlist(purrr::map(tidy_AD_res, ~as.vector(.x@x)))
    CE_posterior_output <- tidy_CE_res@x
    EE_posterior_output <- tidy_EE_res@x
    posterior_output <- c(AD_posterior_output, CE_posterior_output, EE_posterior_output)

    # Use a sensible step size. Given the price elasticity is in the range -1 to 1,
    # we try to limit the step such that the expected change is at the scale of 0.1
    # per step.
    if (missing(step_size)) {
        if (max(abs(variation)) > 0.1) {
            step_size <- 0.1 / max(abs(variation))
        } else {
            step_size <- 1
        }
    }
    next_step <- one_step(AD_prior_input, gradient, size = step_size * direction)

    # The output should be consistent with the input
    list(
        param = AD_prior_input,
        value = AD_res,
        tidy_value = posterior_output,
        Jacobian = Jacobian,
        pca_gradient = gradient,
        next_param = next_step,
        step_size = step_size,
        direction = direction
    )
}

one_step <- function(x, gradient, size) {
    x + size * gradient
}


# Functions to summarise the trajectories --------------------------------------
list_getter <- function(key, simplify = FALSE) {
    function(xs) {
        res <- lapply(xs, \(x) x[[key]])
        if (simplify) return(do.call(cbind, res))
        res
    }
}

get_trajectory_domain <- function(trajectory) {
    # 18 mean and 18 (log) variance parameters
    get_param <- list_getter("param", simplify = TRUE)
    cbind(trajectory$current$param,
          get_param(trajectory$up_path),
          get_param(trajectory$down_path))
}

get_trajectory_range <- function(trajectory) {
    # 34 coefficients + 25 CE + 5 EE
    get_tidy_value <- list_getter("tidy_value", simplify = TRUE)
    res <- cbind(trajectory$current$tidy_value,
                 get_tidy_value(trajectory$up_path),
                 get_tidy_value(trajectory$down_path))
    rownames(res) <- c(rownames(res)[1:34],
                       rownames(trajectory$current$Jacobian))
    res
}

get_trajectory_jacobian <- function(trajectory) {
    # 25 CE + 5 EE wrt 18 mean and (log) variance
    get_jacobian <- list_getter("Jacobian")
    c(list(trajectory$current$Jacobian),
      get_jacobian(trajectory$up_path),
      get_jacobian(trajectory$down_path))
}
