# Get a gradient from a Jacobian matrix
most_sensitive_direction <- function(x) {
    pca <- prcomp(x, center = T, scale. = T)
    weights <- pca$rotation[, 1]
    list(gradient = weights, variation = x %*% weights)
}

`%o%` <- purrr::compose


# Compute the most influential input parameters and the most sensitive output parameters
which_max_k <- function(x, k = 3, value = FALSE) {
    res <- tail(sort(x), k)
    if (value) res else names(res)
}

which_influential_k <- function(x, k, subset = 1:36) {
    which_max_k(
        # take absolute inside for aggregate "displacement",
        # take absolute outside to compare magnitude
        abs(colSums(abs(rename_jacobian_columns(x)[, subset]))),
        k = k, value = FALSE)
}

which_sensitive_k <- function(x, k, subset = 1:36) {
    which_max_k(
        # take absolute inside for aggregate "displacement",
        # take absolute outside to compare magnitude
        abs(rowSums(abs(rename_jacobian_rows(x)[, subset]))),
        k = k, value = FALSE)
}


# Functions working with high sensitivity region ===============================
get_high_sensitivity_region <- function(trajectory_domain, influential_input) {
    x <- trajectory_domain |> t() |> rename_jacobian_columns() |> t()
    t(x[influential_input, ])
}

generate_grid_from_bound <- function(x, k = 2, max_dim = 20) {
    if (ncol(x) > max_dim) {
        stop("You have more dimensions than the maximum dimension allowed. Use a higher number for `max_dim` if you wish to go further.")
    }
    1:ncol(x) |>
        lapply(function(ind) {
            r <- range(x[, ind])
            seq(r[1], r[2], length.out = k)
        }) |>
        expand.grid() |>
        set_colnames(colnames(x))
}


# Extract the prior mean location parameters of the high sensitivity region and
# pair them with their respective precision parameters.
# The current implementation assumes the input are all location parameters.
mean_to_pairs <- function(high_sensitivity_region, domain_bound) {
   lapply(colnames(high_sensitivity_region),
          function(input_key) {
              stopifnot(grepl(pattern = "mean", input_key, fixed = TRUE))
              xy_names <- c(input_key, gsub("mean", "logvar", input_key))
              xy_index <- sapply(xy_names, colnames_to_index)

              res <- t(domain_bound[xy_index, ])
              colnames(res) <- xy_names
              res
          })
}


# Compute the variation of the manifold
manifold_variation <- function(manifold) {
    new_X <- 1:10 %>%
        lapply(\(i) manifold$sample_fun()) %>%
        do.call(rbind, .)
    new_y <- predict_gp(manifold$model, new_X)
    res <- range(new_y$mean)
    c(inf = res[1], sup = res[2], variation = diff(res))
}


# Replace variance upper bound
replace_variance_UB <- function(bound, freq_est, ratio = 1 / 9) {
    freq_variance <- setup_hyperparameter(freq_est, ratio)$PriorCov
    freq_log_variance <- log(diag(freq_variance))

    var_pred <- grepl(colnames(bound), pattern = "logvar")
    if (any(var_pred)) {
        for (i in which(var_pred)) {
            var_key <- colnames(bound)[i]
            var_index <- colnames_to_index(var_key) - 18
            new_variance <- freq_log_variance[var_index]
            bound[2, i] <- new_variance
        }
    }
    bound
}
