#' Marshallian own-price elasticity
#'
#' @description Compute the Marshallian cross-price elasticity of good i
#' with respect itself.
#'
#' @param g_ii A number; the price-effect coefficient of good i on itself.
#' @param b_i A number; the income effect on good i.
#' @param w_i A vector; the budget share / weights for good i.
#'
#' @return A number; the Marshallian own-price elasticity.
#' @export
OwnElast <- function(g_ii, b_i, w_i) {
    CrossElast(g_ii, b_i, w_i, w_i) - 1
}

#' Marshallian cross-price elasticity
#'
#' @description Compute the Marshallian cross-price elasticity of good i
#' with respect to good j.
#'
#' @param g_ij A number; the price-effect coefficient of good j on good i.
#' @param b_i A number; the income effect on good i.
#' @param w_j A vector; the budget share / weights for good j.
#' @param w_i A vector; the budget share / weights for good i.
#'
#' @return A number; the Marshallian cross-price elasticity.
#' @export
CrossElast <- function(g_ij, b_i, w_j, w_i) {
    w_bar_j <- mean(w_j)
    w_bar_i <- mean(w_i)
    (g_ij - b_i * w_bar_j) / w_bar_i
}


#' Income expenditure elasticity
#'
#' @param b_i A number; the income effect on good i.
#' @param w_i A vector; the budget share / weights for good i.
#'
#' @return A number; the income expenditure elasticity.
#' @export
ExpenditureElast <- function(b_i, w_i) {
    w_bar_i <- mean(w_i)
    b_i / w_bar_i + 1
}


#' Hicksian own-price elasticity
#'
#' @param g_ii A number; the price-effect coefficient of good i on itself.
#' @param b_i A number; the income effect on good i.
#' @param w_i A vector; the budget share / weights for good i.
#'
#' @return A number; the Hicksian own-price elasticity.
Hicksian_OwnElast <- function(g_ii, b_i, w_i) {
    w_bar_i <- mean(w_i)
    OwnElast(g_ii, b_i, w_i) + w_bar_i * ExpenditureElast(b_i, w_i)
}

#' Hicksian cross-price elasticity
#'
#' @param g_ij A number; the price-effect coefficient of good j on good i.
#' @param b_i A number; the income effect on good i.
#' @param w_j A vector; the budget share / weights for good j.
#' @param w_i A vector; the budget share / weights for good i.
#'
#' @return A number; the Hicksian cross-price elasticity.
Hicksian_CrossElast <- function(g_ij, b_i, w_j, w_i) {
    w_bar_j <- mean(w_j)
    CrossElast(g_ij, b_i, w_j, w_i) + w_bar_j * ExpenditureElast(b_i, w_i)
}
