get_Jacobian_EE <- function(tidy_EE) {
    set_rownames(tidy_EE@dx, paste("EE", 1:5, sep = "_"))
}

get_Jacobian_CE <- function(tidy_CE) {
    CE_labels <- expand.grid(1:5,1:5) %>%
        apply(1, function(x) paste(x, collapse = "")) %>%
        paste0("CE_", .)
    set_rownames(tidy_CE@dx, CE_labels)
}

tidy_AD_bayes_EE <- function(EE, stat = colMeans) stat(EE)

tidy_AD_bayes_CE <- function(CE, stat = mean) {
    1:5 %>%
        purrr::map(function(row_index) {
            CE[purrr::map_lgl(CE, ~.x$i == row_index)] %>%
                purrr::map(~as.matrix(stat(.x$c_ij))) %>%
                purrr::reduce(cbind)
        }) %>%
        purrr::reduce(rbind)
}


# Interfaces for computing Cross-Price Elasticity ------------------------------
#' Bayesian estimate of Expenditure Elasticity
#' @param bayes_est An object returned by `Gibbs_sampler`.
#' @param data0 The source data.
#' @keywords internal
# AD_bayes_EE :: [delta: [duals], Sigma: [duals]]
AD_bayes_EE <- function(bayes_est, data0) {
    bayes_est <- bayes_est$delta %>%
        {t(purrr::reduce(., cbind))} %>%
        AD_partial_to_full()
    AD_ExpenditureElast_vec(bayes_est, data0)
}

#' Compute the Expenditure Elasticity
#' @param coef0 A dataframe of full estimated coefficients.
#' @param data0 The data
#' @keywords internal
AD_ExpenditureElast_vec <- function(coef0, data0) {
    1:5 %>%
        purrr::map(function(i) {
            w_i <- data0[[paste0("w", i)]]
            b_i <- coef0[, name_to_full_column_index(paste0("b_", i))]
            as.matrix(ExpenditureElast(b_i, w_i))
        }) %>%
        purrr::reduce(cbind)
}


# Interfaces for computing Cross-Price Elasticity ------------------------------
#' Bayesian estimate of Cross-Price Elasticity
#' @param bayes_est An object returned by `Gibbs_sampler`.
#' @param data0 The source data.
#' @keywords internal
# AD_bayes_CE :: [delta: [duals], Sigma: [duals]]
AD_bayes_CE <- function(bayes_est, data0) {
    bayes_est <- bayes_est$delta %>%
        {t(purrr::reduce(., cbind))} %>%
        AD_partial_to_full()
    AD_CrossElast_matrix(bayes_est, data0)
}

AD_CrossElast_matrix <- function(coef0, data0) {
    purrr::map(1:5, function(i) {
        purrr::map(1:5, function(j) {
            g_ij <- coef0[, name_to_full_column_index(paste0("g_", i, j))]
            b_i <- coef0[, name_to_full_column_index(paste0("b_", i))]
            w_j <- data0[[paste0("w", j)]]
            w_i <- data0[[paste0("w", i)]]
            if (i == j) {
                list(i = i, j = j, c_ij = OwnElast(g_ij, b_i, w_i))
            } else {
                list(i = i, j = j, c_ij = CrossElast(g_ij, b_i, w_j, w_i))
            }
        })
    }) %>%
        purrr::reduce(append)
}


# Share utility ----------------------------------------------------------------
name_to_full_column_index <- function(x) {
    names_holder <- as.data.frame(t(1:18)) %>%
        set_colnames(colnames(rename(1:18))) %>%
        partial_to_full() %>%
        colnames()
    purrr::map_dbl(x, ~which(names_holder == .x))
}

#' Maps the reduced set of parameters back to the unconstrained set of parameters
#' @param param A dataframe; each row contains the coefficients.
#' @keywords internal
AD_partial_to_full <- function(param) {
    f <- Vectorize(function(x) which(colnames(rename(1:18)) == x))
    new_param <- list()
    new_param$a_5 <- 1 - param[, f("a_1"), drop = F] - param[, f("a_2"), drop = F] - param[, f("a_3"), drop = F] - param[, f("a_4"), drop = F]
    # entry_ij with i > j (use symmetry)
    new_param$g_21 <- param[, f("g_12"), drop = F]
    new_param$g_31 <- param[, f("g_13"), drop = F]
    new_param$g_32 <- param[, f("g_23"), drop = F]
    new_param$g_41 <- param[, f("g_14"), drop = F]
    new_param$g_42 <- param[, f("g_24"), drop = F]
    new_param$g_43 <- param[, f("g_34"), drop = F]
    # entry ending with index 5 (use coefficient constraints)
    new_param$g_15 <- - param[, f("g_11")] - param[, f("g_12")] - param[, f("g_13")] - param[, f("g_14"), drop = F]
    new_param$g_25 <- - new_param$g_21     - param[, f("g_22")] - param[, f("g_23")] - param[, f("g_24"), drop = F]
    new_param$g_35 <- - new_param$g_31     - new_param$g_32     - param[, f("g_33")] - param[, f("g_34"), drop = F]
    new_param$g_45 <- - new_param$g_41     - new_param$g_42     - new_param$g_43     - param[, f("g_44"), drop = F]
    # entry starting with index 5 (use coefficient constraints)
    new_param$g_51 <- new_param$g_15
    new_param$g_52 <- new_param$g_25
    new_param$g_53 <- new_param$g_35
    new_param$g_54 <- new_param$g_45
    new_param$g_55 <- - new_param$g_51 - new_param$g_52 - new_param$g_53 - new_param$g_54
    new_param$b_5 <- - param[, f("b_1")] - param[, f("b_2")] - param[, f("b_3")] - param[, f("b_4"), drop = F]
    cbind(
        param[, f(c("a_1", "a_2", "a_3", "a_4"))], new_param$a_5,
        param[, f(c("b_1", "b_2", "b_3", "b_4"))], new_param$b_5,
        param[, f(c("g_11", "g_12", "g_13", "g_14"))], new_param$g_15,
        new_param$g_21, param[, f(c("g_22", "g_23", "g_24"))], new_param$g_25,
        new_param$g_31, new_param$g_32, param[, f(c("g_33", "g_34"))], new_param$g_35,
        new_param$g_41, new_param$g_42, new_param$g_43, param[, f(c("g_44")), drop = F], new_param$g_45,
        new_param$g_51, new_param$g_52, new_param$g_53, new_param$g_54, new_param$g_55
    )
}

