#' Post-process the Posterior Expenditure Elasticity samples from the Gibbs sampler
#' @param bayes_EE An object returned by `bayes_EE`
#' @param burn_in The number of burn-ins; default to be one-fifth of the total runs.
#' @param fun A function, for computing the posterior statistics.
#' @export
tidy_bayes_EE <- function(bayes_EE, burn_in, fun) {
    if (missing(burn_in)) burn_in <- round(nrow(bayes_EE) / 10)
    apply(bayes_EE[(burn_in + 1):nrow(bayes_EE), ], 2, fun)
}

#' Post-process the Posterior Cross-price Elasticity samples from the Gibbs sampler
#' @param bayes_CE An object returned by `bayes_CE`
#' @param burn_in The number of burn-ins; default to be one-fifth of the total runs.
#' @param fun A function, for computing the posterior statistics.
#' @export
tidy_bayes_CE <- function(bayes_CE, burn_in, fun) {
    n <- dim(bayes_CE)[3]
    if (missing(burn_in)) burn_in <- round(n / 10)
    apply(bayes_CE[,, (burn_in + 1):n], c(1, 2), fun)
}


# Interfaces for computing Expenditure Elasticity --------------------------------
#' Bayesian estimate of Expenditure Elasticity
#' @param bayes_est An object returned by `Gibbs_sampler`.
#' @param data0 The source data.
#' @export
bayes_EE <- function(bayes_est, data0) {
    data.frame(bayes_est$delta) %>%
        set_colnames(colnames(rename(1:18))) %>%
        partial_to_full() %>%
        ExpenditureElast_vec(data0)
}

#' Frequentist estimate of Expenditure Elasticity
#' @param freq_est An object returned by `frequentist_estimate`.
#' @param data0 The source data.
#' @export
freq_EE <- function(freq_est, data0) {
    tidy_freq_est(freq_est) %>%
        partial_to_full() %>%
        ExpenditureElast_vec(data0) %>%
        as.numeric()
}

#' Prior specification of Expenditure Elasticity
#' @param prior_spec An object returned by `setup_hyperparameter`.
#' @param data0 The source data.
#' @export
prior_EE <- function(prior_spec, data0) {
    data.frame(t(prior_spec$PriorMean)) %>%
        partial_to_full() %>%
        ExpenditureElast_vec(data0) %>%
        as.numeric()
}


#' Compute the Expenditure Elasticity
#' @param coef0 A data frame of full estimated coefficients.
#' @param data0 The data
ExpenditureElast_vec <- function(coef0, data0) {
    res <- c()
    for (i in 1:5) {
        w_i <- data0[[paste0("w", i)]]
        b_i <- coef0[, paste0("b_", i)]
        res <- cbind(res, ExpenditureElast(b_i, w_i))
    }
    res
}


# Interfaces for computing Cross-Price Elasticity --------------------------------
#' Bayesian estimate of Cross-Price Elasticity
#' @param bayes_est An object returned by `Gibbs_sampler`.
#' @param data0 The source data.
#' @export
bayes_CE <- function(bayes_est, data0) {
    data.frame(bayes_est$delta) %>%
        set_colnames(colnames(rename(1:18))) %>%
        partial_to_full() %>%
        CrossElast_matrix(data0)
}

#' Frequentist estimate of Cross-Price Elasticity
#' @param freq_est An object returned by `frequentist_estimate`.
#' @param data0 The source data.
#' @export
freq_CE <- function(freq_est, data0) {
    tidy_freq_est(freq_est) %>%
        partial_to_full() %>%
        CrossElast_matrix(data0) %>%
        magrittr::extract(, , 1)
}

#' Prior specification of Cross-Price Elasticity
#' @param prior_spec An object returned by `setup_hyperparameter`.
#' @param data0 The source data.
#' @export
prior_CE <- function(prior_spec, data0) {
    data.frame(t(prior_spec$PriorMean)) %>%
        partial_to_full() %>%
        CrossElast_matrix(data0) %>%
        magrittr::extract(, , 1)
}


#' Compute the Cross-Price Elasticity
#' @param coef0 A data frame of full estimated coefficients.
#' @param data0 The data
CrossElast_matrix <- function(coef0, data0) {
    Mar_PE <- array(0, c(5, 5, nrow(coef0)))
    for (i in 1:5) {
        for (j in 1:5) {
            g_ij <- coef0[, paste0("g_", i, j)]
            b_i <- coef0[, paste0("b_", i)]
            w_j <- data0[[paste0("w", j)]]
            w_i <- data0[[paste0("w", i)]]
            if (i == j) {
                Mar_PE[i, j, ] <- OwnElast(g_ij, b_i, w_i)
            } else {
                Mar_PE[i, j, ] <- CrossElast(g_ij, b_i, w_j, w_i)
            }
        }
    }
    Mar_PE
}


#' Maps the reduced set of parameters back to the unconstrained set of parameters
#' @param param A data frame, of which each column contains the coefficients.
partial_to_full <- function(param) {
    param$a_5 <- 1 - param$a_1 - param$a_2 - param$a_3 - param$a_4
    # entry_ij with i > j (use symmetry)
    param$g_21 <- param$g_12
    param$g_31 <- param$g_13
    param$g_32 <- param$g_23
    param$g_41 <- param$g_14
    param$g_42 <- param$g_24
    param$g_43 <- param$g_34
    # entry ending with index 5 (use coefficient constraints)
    param$g_15 <- - param$g_11 - param$g_12 - param$g_13 - param$g_14
    param$g_25 <- - param$g_21 - param$g_22 - param$g_23 - param$g_24
    param$g_35 <- - param$g_31 - param$g_32 - param$g_33 - param$g_34
    param$g_45 <- - param$g_41 - param$g_42 - param$g_43 - param$g_44
    # entry starting with index 5 (use coefficient constraints)
    param$g_51 <- param$g_15
    param$g_52 <- param$g_25
    param$g_53 <- param$g_35
    param$g_54 <- param$g_45
    param$g_55 <- - param$g_51 - param$g_52 - param$g_53 - param$g_54
    param$b_5 <- - param$b_1 - param$b_2 - param$b_3 - param$b_4
    param[order(names(param))]
}
