#' Title
#'
#' @param ce_jacobian The Jacobian matrix of the cross-price elasticities.
#' @param ee_jacobian The Jacobian matrix of the expenditure elasticities.
#' @param suffix A suffix for the output filename. Useful to identify which prior
#' is used for the calculation.
#' @param color_symmetry_cap TRUE or FALSE; whether to cap the values so that
#' the color scale is symmetry. Asymmetric color scale can sometimes be misleading.
#' @param save_to_file TRUE or FALSE; whether to save the file. If FALSE, the
#' plot function is called without the `png` device setup call.
#' @param scale_of_variance A number; preferably the powers of ten. This is needed
#' for `corrplot` to display values on the color scale properly. The issue is
#' that small numbers may appear as zero in the scale due to lack of space. This
#' happens with the variance parameter in particular.
#' @return A Jacobian matrix corresponding to the plot.
#' @export
plot_jacobian <- function(ce_jacobian, ee_jacobian, suffix,
                          color_symmetry_cap = TRUE,
                          save_to_file = TRUE,
                          scale_of_variance = 1) {
    # Use math labels for the plots
    param_labels <- c("alpha[1]", "alpha[2]", "alpha[3]", "alpha[4]", "gamma[11]", "gamma[12]", "gamma[13]", "gamma[14]", "gamma[22]",
                      "gamma[23]", "gamma[24]", "gamma[33]", "gamma[34]", "gamma[44]", "beta[1]", "beta[2]", "beta[3]", "beta[4]")
    param_labels <- paste0(":", param_labels)

    ce_labels <- rownames(ce_jacobian) %>%
        purrr::map_chr(~unlist(strsplit(.x, "_"))[[2]]) %>%
        paste0(":epsilon[", ., "]")
    ee_labels <- paste0(":eta[", 1:5, "]")

    # Expenditure elasticities
    if (save_to_file) {
        filename <- output(paste0("figures/Jacobian_price_elasticities_", suffix, ".png"))
        png(file = filename, width = 1000, height = 560)
    } else {
        message("Note that `par` default has been modified. Use `dev.off()` to restore the default.")
    }

    par(mfrow = c(1,2))

    rownames(ce_jacobian) <- ce_labels
    colnames(ce_jacobian) <- c(param_labels, param_labels)
    ce_jacobian <- ce_jacobian[-c(seq(5, 25, 5), 20:24), ]  # show 4 groups

    font_size <- 1.5
    # corr_plot(ce_jacobian[, 1:18])(tl.cex = font_size, cl.cex = font_size)   # wrt prior mean
    # corr_plot(ce_jacobian[, 19:36])(tl.cex = font_size, cl.cex = font_size)  # wrt prior variance, scaled to have a nicer legend
    # dev.off()

    # Cross price elasticities
    # filename <- paste0("figures/Jacobian_ee_", name, "_rho_", confidence, ".png")
    # png(file = filename, width = 1200, height = 375)
    # par(mfrow = c(1,2))
    #
    rownames(ee_jacobian) <- ee_labels
    colnames(ee_jacobian) <- rep("", 36)
    ee_jacobian <- ee_jacobian[1:4, ]

    # corr_plot(ee_jacobian[, 1:18])(tl.cex = font_size, cl.cex = font_size)   # wrt prior mean
    # corr_plot(ee_jacobian[, 19:36])(tl.cex = font_size, cl.cex = font_size)  # wrt prior variance, scaled to have a nicer legend
    # dev.off()
    # par(mfrow = c(1,1))

    jacobian <- rbind(ce_jacobian, ee_jacobian)
    mean_jacobian <- jacobian[, 1:18]
    logvar_jacobian <- jacobian[, 18 + (1:18)]

    # jacobian <- pmin(jacobian, 1.51)
    if (color_symmetry_cap) {
        clip_value <- function(x) {
            rng <- abs(range(x))
            clip_positive <- diff(rng) > 0
            if (clip_positive) {
                x <- pmin(x, rng[1])
            } else {
                x <- pmax(x, -rng[2])
            }
            x
        }
        mean_jacobian <- clip_value(mean_jacobian)
        logvar_jacobian <- clip_value(logvar_jacobian)
    }

    corr_plot(mean_jacobian)(tl.cex=font_size, cl.cex=font_size)   # wrt prior mean
    mtext("Prior mean", at=10, line=2, cex=font_size * 1.3, family="DejaVu Sans", font=2)

    corr_plot(logvar_jacobian / scale_of_variance)(tl.cex=font_size, cl.cex=font_size)  # wrt prior variance, scaled to have a nicer legend
    mtext("Log of prior variance", at=10, line=2, cex=font_size * 1.3, family="DejaVu Sans", font=2)
    if (scale_of_variance != 1) {
        mtext(TeX_scale(scale_of_variance),
              at=20.05, line=1, cex=font_size, family="DejaVu Sans", font=2)
    }

    if (save_to_file) {
        dev.off()
    }
    jacobian
}

TeX_scale <- function(x) {
    TeX(paste0("$\\times 10^{", log(x, 10), "}$"))
}

# # Testing
# plot_jacobian(tar_read(bayes_est_CE_AD_Jacobian),
#               tar_read(bayes_est_EE_AD_Jacobian),
#               color_symmetry_cap = FALSE,
#               save_to_file = FALSE)
#
#===============================================================================
plot_aggregate_jacobian <- function(ce_ee_jacobian, suffix,
                                    color_symmetry_cap = TRUE,
                                    save_to_file = TRUE,
                                    scale_of_variance = 1) {
    # Use math labels for the plots
    param_labels <- c("alpha[1]", "alpha[2]", "alpha[3]", "alpha[4]", "gamma[11]", "gamma[12]", "gamma[13]", "gamma[14]", "gamma[22]",
                      "gamma[23]", "gamma[24]", "gamma[33]", "gamma[34]", "gamma[44]", "beta[1]", "beta[2]", "beta[3]", "beta[4]")
    param_labels <- paste0(":", param_labels)

    ce_labels <- rownames(ce_ee_jacobian)[1:25] %>%
        purrr::map_chr(~unlist(strsplit(.x, "_"))[[2]]) %>%
        paste0(":epsilon[", ., "]")
    ee_labels <- paste0(":eta[", 1:5, "]")

    # Expenditure elasticities
    if (save_to_file) {
        filename <- output(paste0("figures/aggregate_Jacobian_price_elasticities_", suffix, ".png"))
        png(file = filename, width = 1000, height = 560)
    } else {
        message("Note that `par` default has been modified. Use `dev.off()` to restore the default.")
    }

    par(mfrow = c(1,2))
    font_size <- 1.5
    rownames(ce_ee_jacobian) <- c(ce_labels, ee_labels)
    colnames(ce_ee_jacobian) <- c(param_labels, param_labels)

    jacobian <- ce_ee_jacobian[-c(seq(5, 25, 5), 20:24, 30), ]
    mean_jacobian <- jacobian[, 1:18]
    logvar_jacobian <- jacobian[, 18 + (1:18)]

    if (color_symmetry_cap) {
        clip_value <- function(x) {
            rng <- abs(range(x))
            clip_positive <- diff(rng) > 0
            if (clip_positive) {
                x <- pmin(x, rng[1])
            } else {
                x <- pmax(x, -rng[2])
            }
            x
        }
        mean_jacobian <- clip_value(mean_jacobian)
        logvar_jacobian <- clip_value(logvar_jacobian)
    }

    corr_plot(mean_jacobian)(tl.cex=font_size, cl.cex=font_size)   # wrt prior mean
    mtext("Prior mean", at=10, line=2, cex=font_size * 1.3, family="DejaVu Sans", font=2)

    corr_plot(logvar_jacobian / scale_of_variance)(tl.cex=font_size, cl.cex=font_size)  # wrt prior variance, scaled to have a nicer legend
    mtext("Log of prior variance", at=10, line=2, cex=font_size * 1.3, family="DejaVu Sans", font=2)
    if (scale_of_variance != 1) {
        mtext(TeX_scale(scale_of_variance),
              at=20.05, line=1, cex=font_size, family="DejaVu Sans", font=2)
    }

    if (save_to_file) {
        dev.off()
    }
    jacobian
}

clip_columns <- function(x, value, subset) {
    stopifnot(value >= 0)
    if (missing(subset)) subset <- 1:ncol(x)
    x[, subset] <- x[, subset] |> pmin(value) |> pmax(-value)
    x
}

math_symbols_to_names <- function(x) {
    x |>
        gsub(pattern = "[:]", replacement = "") |>
        gsub(pattern = "[]]", replacement = "") |>
        gsub(pattern = "[[]", replacement = "_") |>
        gsub(pattern = "gamma", replacement = "g") |>
        gsub(pattern = "beta", replacement = "b") |>
        gsub(pattern = "alpha", replacement = "a") |>
        gsub(pattern = "epsilon", replacement = "CE") |>
        gsub(pattern = "eta", replacement = "EE")
}

rename_jacobian_columns <- function(x) {
    coln <- math_symbols_to_names(colnames(x))
    colnames(x) <- c(paste0("mean-", coln[1:18]),
                     paste0("logvar-", coln[19:36]))
    x
}

rename_jacobian_rows <- function(x) {
    rownames(x) <- math_symbols_to_names(rownames(x))
    x
}

colnames_to_index <- function(xs) {
    ref <- names(c(rename(1:18, "mean-"),
                   rename(1:18, "logvar-")))
    sapply(xs, \(x) which(ref == x), USE.NAMES = FALSE)
}
