#' Set up the prior mean and covariance of the LAID model
#'
#' @description Use the expert prior provided in the package and set
#' the prior covariance by matching the coefficient of variations from
#' the frequentist estimates according to a fixed ratio.
#'
#' @param sysfit_result A returned object from `frequentist_estimate`.
#' @param ratio A positive number; the data-prior ratio.
#' When r = 1, equal weights are placed on data and prior.
#' When r = 0.5, prior has half the weights of data.
#' When r = 2, prior has twice the weights of data.
#'
#' @export
setup_hyperparameter <- function(sysfit_result, ratio = 1) {
    res <- sysfit_result
    coef <- res$coefficients
    sdcoef <- diag(res$coefCov)^0.5
    CV <- coef / sdcoef

    # Reorder coefficients since there is an order mismatch between how the estimates
    # are presented and how we use them
    get_ind <- Vectorize(function(x) which(x == names(coef)))
    labels_ord <- c(
        "eq1_(Intercept)", "eq2_(Intercept)", "eq3_(Intercept)", "eq4_(Intercept)",
        "eq1_lnp1P5", "eq1_lnp2P5", "eq1_lnp3P5", "eq1_lnp4P5",
        "eq2_lnp2P5", "eq2_lnp3P5", "eq2_lnp4P5",
        "eq3_lnp3P5", "eq3_lnp4P5",
        "eq4_lnp4P5",
        "eq1_lnXR", "eq2_lnXR", "eq3_lnXR", "eq4_lnXR"
    )
    location <- get_ind(labels_ord)
    coefOrder <- coef[location]
    sdcoefOrder <- sdcoef[location]
    CVOrder <- c(CV[location])
    CoefFreq <- cbind(coefOrder, sdcoefOrder, CVOrder)

    file <- dataset("HyperMeanLocation.csv")
    HyperMeanLoc <- read.table(file, header = T, sep = ",")[, 2]
    Means <- cbind(coefOrder, HyperMeanLoc)
    s <- (abs(HyperMeanLoc) / (ratio * abs(CVOrder)))^2
    PriorCov <- diag(s)

    variableName <- c("a_1", "a_2", "a_3", "a_4",
                      "g_11", "g_12", "g_13", "g_14", "g_22",
                      "g_23", "g_24", "g_33", "g_34", "g_44",
                      "b_1", "b_2", "b_3", "b_4")
    names(HyperMeanLoc) <- variableName
    rownames(PriorCov) <- colnames(PriorCov) <- variableName
    list(PriorMean = HyperMeanLoc, PriorCov = PriorCov)
}


#' Set up the prior mean and covariance of the LAID model
#'
#' @description Use the non-informative prior mean and set
#' the prior covariance by matching the coefficient of variations from
#' the frequentist estimates according to a fixed ratio.
#'
#' @param sysfit_result A returned object from `frequentist_estimate`.
#' @param ratio A positive number; the data-prior ratio.
#' When r = 1, equal weights are placed on data and prior.
#' When r = 0.5, prior has half the weights of data.
#' When r = 2, prior has twice the weights of data.
#'
#' @export
zero_prior <- function(sysfit_result, ratio = 1) {
    priors <- setup_hyperparameter(sysfit_result, ratio)
    priors$PriorMean <- rep(0, 18) %>%
        setNames(names(priors$PriorMean))
    priors
}


#' Convert confidence weights on prior into data-prior ratios
#'
#' @param weights A number between 0 and 1 (exclusively); the weight / confidence
#' on the prior, with 0 representing no confidence and 1 representing certainty.
#'
#' @export
convert_weights_to_ratios <- function(weights) {
    sapply(weights, function(weight) 1 / (1 - weight) - 1)
}


#' Linearly interpolate between two sets prior specifications
#'
#' @param weight A number between 0 and 1; the weight on the first prior.
#' @param prior_1 A prior specification.
#' @param prior_2 A prior specification.
#' @param log_variance TRUE or FALSE; whether the variance should be interpolated
#' in the log-space.
#'
#' @export
linear_interpolate <- function(weight, prior_1, prior_2, log = TRUE) {
    if (log) {
        prior_1 <- log_variance(prior_1)
        prior_2 <- log_variance(prior_2)
    }

    new_prior <- lapply(names(prior_1),
                        function(key) prior_1[[key]] * weight + (1 - weight) * prior_2[[key]])
    names(new_prior) <- names(prior_1)

    if (log) {
        return(exp_variance(new_prior))
    }
    new_prior
}

log_variance <- function(prior) {
    prior$PriorCov <- log(prior$PriorCov)
    prior
}

exp_variance <- function(prior) {
    prior$PriorCov <- exp(prior$PriorCov)
    prior
}
