#' Inference of the LAID model using Gibbs Sampler
#'
#' @description Perform estimation and compute the posterior statistics of
#' the coefficients and price elasticities.
#'
#' @param prior0 A named list with names "PriorMean" and "PriorCov".
#' @param data0 The data.
#' @param seed Seed for random number generator.
#' @param burn_in Number of samplers to discard as burn-ins.
#' @param ... Optional argument to pass to \link{Gibbs_sampler}.
#'
#' @export
Gibbs_sampler_pipe <- function(prior0, data0, seed = 123,
                               burn_in = 100, ...) {
    set.seed(seed)   # ensure path-wise continuity
    bayes_samples <- Gibbs_sampler(
        data = data0,
        d0 = prior0$PriorMean, D0 = prior0$PriorCov,
        ...
    )
    bayes_EE_samples = bayes_EE(bayes_samples, data0)
    bayes_CE_samples = bayes_CE(bayes_samples, data0)

    # Compute the posterior statistics ----
    bayes_est = concat_list(
        mcmc_slicing(bayes_samples$delta, burn_ins = burn_in),
        mcmc_slicing(bayes_samples$Sigma, burn_ins = burn_in)
    )
    bayes_exp_elast = mcmc_slicing(bayes_EE_samples, burn_ins = burn_in)
    bayes_cross_elast = mcmc_slicing(bayes_CE_samples, burn_ins = burn_in, along = 3)

    list(
        bayes_est = bayes_est,
        bayes_exp_elast = bayes_exp_elast,
        bayes_cross_elast = bayes_cross_elast
    )
}

mcmc_slicing <- function(draws, burn_ins, block_size = 50, along = 1) {
    if (missing(burn_ins)) {
        burn_ins <- round(NROW(draws) / 10)
    }
    draws <- slice_samples(draws, -(1:burn_ins), along = along)

    sliced_indices <- block_indices(block_size, dim(draws)[along])
    sliced_samples <- lapply(sliced_indices, \(ind) slice_samples(draws, ind, along))
    sliced_means <- abind::abind(lapply(sliced_samples, block_mean, along = along),
                                 along = 0)
    list(mean = apply2(sliced_means, mean),
         se = apply2(sliced_means, sd))
}

block_indices <- function(block_size, total) {
    lapply(1:block_size,
           function(start) seq(start, total, by = block_size))
}

block_mean <- function(block, ...) apply2(block, mean, ...)

slice_samples <- function(x, ind, along = 1) {
    args <- lapply(1:(length(dim(x))+1), function(index) {
        if (index == 1) return(x)
        if (index == along + 1) return(ind)
        quote(expr=)
    })
    do.call(`[`, args)
}

apply2 <- function(x, f, along = 1) {
    margin <- seq_along(dim(x))[-along]
    apply(x, margin, f)
}


concat_list <- function(x, y) {
    result <- lapply(names(x), function(key) c(x[[key]], y[[key]]))
    names(result) <- names(x)
    result
}


#' Evaluating a point in the prior parameter space
#' @note This function is created for active learning
#'
#' @param prior_parameters The parameters that are different to the reference parameters.
#' The variance should be in log scale.
#' @param prior_skeleton The reference full parameter list.
#' @param key The output variable to return; one of 'bayes_est', 'bayes_exp_elast', 'bayes_cross_elast'.
#' @param index The index of the target entry.
#'
#' @return A scalar (the posterior mean) with the standard error in the attribute 'sigma'.
#'
#' @export
mcmc_evaluation <- function(prior_parameters, prior_skeleton, data0, key, index) {
    # Use a vector to facilitate replacement of values
    new_priors <- prior_skeleton
    param_vector <- c(new_priors$PriorMean, log(diag(new_priors$PriorCov)))

    ind <- colnames_to_index(names(prior_parameters))
    param_vector[ind] <- prior_parameters
    param <- list(PriorMean = param_vector[1:18],
                  PriorCov = diag(exp(param_vector[19:36])))

    res <- Gibbs_sampler_pipe(param, data = data0, num_iter = 1e3 + 1e2,
                              burn_in = 1e2, Wv = 4, WEsc0 = diag(4),
                              Sig = diag(4))
    result <- res[[key]]$mean[index]
    attr(result, "sigma") <- res[[key]]$se[index]
    result
}
