#' Post-process the Posterior samples from the Gibbs sampler
#'
#' @param bayes_res An object returned by `Gibbs_sampler`
#' @param burn_in The number of burn-ins; default to be one-fifth of the total runs.
#' @param fun A function, for computing the posterior statistics.
#'
#' @export
tidy_bayes_est <- function(bayes_res, burn_in, fun = mean) {
    if (missing(burn_in)) {
        burn_in <- round(nrow(est) / 10)
    }

    est <- bayes_res$delta
    tidy_delta <- rename(apply(est[(burn_in + 1):nrow(est), ], 2, fun))

    est2 <- bayes_res$Sigma
    tidy_Sigma <- apply(est2[(burn_in + 1):nrow(est), ,], c(2,3), fun)

    c(tidy_delta, tidy_Sigma)
}


#' Inference of the LAID model using Gibbs Sampler
#'
#' @param data A data frame; the source data.
#' @param d0 Prior location parameters (Normal prior).
#' @param D0 Prior covariance (Normal prior).
#' @param Wv Prior degrees of freedom (Wishart prior).
#' @param WEsc0 Prior scale matrix (Wishart prior).
#' @param Sig Starting value of the covariance of the normal prior.
#' @param num_iter Number of MCMC Iterations.
#'
#' @export
Gibbs_sampler <- function(data, d0, D0, Wv, WEsc0, Sig, num_iter = 1e4) {
    Ch <- format_SUR_data(data)
    s <- with(data, cbind(w1, w2, w3, w4))

    d0 <- as.matrix(d0)
    N <- dim(Ch)[3]
    d <- d0
    # Helper variables
    inv_D0 <- solve(D0)
    inv_D0_times_d0 <- inv_D0 %*% d0
    inv_WEsc0 <- solve(WEsc0)
    inv_Sig <- solve(Sig)
    # Storage variables
    delta <- matrix(0, num_iter, 4 + 10 + 4) # 4 a's, 10 g's and 4 b's! #Save location posteior parameters
    Sigma <- array(1, c(num_iter, 4, 4))     # Save covariance matrix
    tic <- Sys.time() # Time

    pb <- txtProgressBar(min = 1, max = num_iter, initial = 1, style = 3)
    for (i in 1:num_iter) {
        AuxSum <- matrix(0, 18, 18)
        AuxSum1 <- matrix(0, 18, 1)
        for (h in 1:N) {
            Ch_h <- Ch[, , h]
            s_h  <- s[h, ]
            t_CH_h_times_inv_Sig <- t(Ch_h) %*% inv_Sig
            AuxSum  <- AuxSum  + t_CH_h_times_inv_Sig %*% Ch_h
            AuxSum1 <- AuxSum1 + t_CH_h_times_inv_Sig %*% s_h
        }

        SigDel <- solve(inv_D0 + AuxSum)
        MeanDel <- SigDel %*% (AuxSum1 + inv_D0_times_d0)
        d <- MeanDel + t(chol(SigDel)) %*% rnorm(length(d0))

        AuxSum2 <- matrix(0, 4, 4)
        for (h in 1:N) {
            s_minus_Ch_times_d <- s[h, ] - Ch[, , h] %*% d
            AuxSum2 <- AuxSum2 + s_minus_Ch_times_d %*% t(s_minus_Ch_times_d)
        }
        WScale <- solve(inv_WEsc0 + AuxSum2)
        vNew <- Wv + N
        inv_Sig <- matrix(rWishart(1, vNew, WScale), 4, 4)
        Sig <- solve(inv_Sig)

        delta[i, ] <- d
        Sigma[i, , ] <- Sig
        setTxtProgressBar(pb, value = i)
    }

    toc <- Sys.time() # Time
    print(toc - tic)  # Total time
    list(delta = delta, Sigma = Sigma)
}

#' Convert data to a format appropriate for the SUR model estimation
#'
#' @param data0 A data frame; the data.
#' @export
format_SUR_data <- function(data0) {
    N <- nrow(data0)
    Ch <- array(0, c(4, 4 + 4 + 3 + 2 + 1 + 4, N))
    I_4 <- diag(4)
    D_4 <- matrixcalc::duplication.matrix(4)
    for (h in 1:N) {
        log_pi <- t(as.numeric(data0[h, 6:9]))  # row vector
        log_XR <- data0[h, 11]
        Ch[, , h] <- cbind(I_4, (I_4 %x% log_pi) %*% D_4, I_4 * log_XR)
    }
    Ch
}
