#' Gaussian Process
#'
#' @param X A numeric matrix; the data.
#' @param y A numeric vector; the data.
#' @param kernel A kernel object, which is a named list of a list of
#' named parameters and an uninitialised kernel function.
#' @param sigma A positive number; the noise of the data.
#' @param options Optional argument to pass to \code{optim}.
#'
#' @examples
#' \dontrun{
#' library(gptools2)
#' x <-  as.matrix(seq(-10, 10, 1.5))
#' f <- sin
#' y <- f(x)
#' model <- gp(x, y, sigma = 0)
#' pred_y <- predict_gp(model, x)
#' cbind(y, pred_y$mean, err = y - pred_y$mean)
#'
#' test_x <- as.matrix(seq(-10, 10, 0.1))
#' test_y <- f(test_x)
#' pred_y <- predict_gp(model, test_x)
#' cbind(test_y, pred_y$mean, err = test_y - pred_y$mean)
#'
#' plot(x, y, pch = 19, xlim = c(-10, 10))
#' lines(test_x, test_y, type = 'l', lty = 2)
#' lines(test_x, pred_y$mean, col = 'blue')
#' }
#' @export
gp <- function(X, y, kernel = squared_exponential(), sigma = 0, options = list()) {
    X <- as.matrix(X)
    y <- as.numeric(y)
    # loglikelihood functions
    # @param param A vector of parameters
    NLL <- function(param) {
        param_named <- relist(param, kernel$param)
        kern_fun <- do.call(kernel$kern_fun, param_named)
        KpS2I <- kcov(X, X, kern_fun) + sigma^2 * diag(nrow(X))
        -mvtnorm::dmvnorm(y, sigma = KpS2I, log = T)
    }
    options$par <- unlist(kernel$param)
    options$fn <- NLL
    parameters <- do.call(optim, options)
    list(
        train_X = X, train_y = y,
        kernel = kernel, sigma = sigma,
        parameters = relist(parameters$par, kernel$param),
        optim_log = parameters
    )
}

#' Compute the covariance matrix
#'
#' @param X1 A numeric matrix.
#' @param X2 A numeric matrix, must have the same number of columns as X1.
#' @param k A function that computes the covariance of two vectors.
#'
#' @return A numeric matrix; the covariance matrix
#' @export
kcov <- function(X1, X2, k) {
    k(kcov_C(X1, X2))
}


#' Compute the Euclidean distance of two vectors
#'
#' @param x1 A numeric vector.
#' @param x2 A numeric vector.
#'
#' @return A positive number.
#' @export
euclidean <- function(x1, x2) {
    norm(x1 - x2, "2")
}


#' Compute the log determinant
#'
#' @param x A numeric matrix
#'
#' @return A numeric number
log_det <- function(x) {
    determinant(x, logarithm = TRUE)$modulus
}


#' Predict using the Gaussian Process
#'
#' @param model Output from \code{gp}; the model object.
#' @param new_X A numeric matrix; the points to predict.
#'
#' @return A named list containing all the input and the
#' predictive mean vector and covariance matrix.
#' @export
predict_gp <- function(model, new_X) {
    X <- model$train_X
    y <- model$train_y
    kern_fun <- do.call(model$kernel$kern_fun, model$parameters)
    sigma <- model$sigma

    K_11 <- kcov(X, X, kern_fun) + sigma^2 * diag(nrow(X))
    K_12 <- kcov(X, new_X, kern_fun)
    K_21 <- t(K_12)
    K_22 <- kcov(new_X, new_X, kern_fun)

    list(
        model = model,
        new_X = new_X,
        mean = K_21 %*% solve(K_11, y),
        covariance = K_22 - K_21 %*% solve(K_11, K_12)
    )
}


#' Kernel template
#'
#' @param param A named list; the parameters of the kernel.
#' @param kern A function; the kernel function that takes the parameters
#' and returns a function that computes the covariance between two input
#' vectors.
#'
#' @return A named list with the "kernel" tag.
kernel_template <- function(param, kern) {
    structure(
        list(param = param, kern_fun = kern),
        class = c("kernel", "list")
    )
}


#' Squared exponential kernel
#'
#' @name se_kernel
#' @param sigma Positive number; the parameter.
#' @param l Positive number; the parameter.
#'
#' @return A kernel object.
#'
#' @export
squared_exponential <- function(sigma = 1, l = 1) {
    kernel_template(
        list(sigma = sigma, l = l),
        function(sigma, l) {
            function(X) {
                sigma^2 * exp(X / (-2 * l^2))
            }
        }
    )
}


#' Rational quadratic kernel
#'
#' @name rq_kernel
#' @param sigma Positive number; the parameter.
#' @param l Positive number; the parameter.
#' @param alpha Positive number; the parameter.
#'
#' @return A kernel object.
#'
#' @export
rational_quadratic_kernel <- function(sigma = 1, l = 1, alpha = 1) {
    kernel_template(
        list(sigma = sigma, l = l, alpha = alpha),
        function(sigma, l, alpha) {
            function(X) {
                sigma^2 * (1 + X / (2 * alpha * l^2)) ^ (-alpha)
            }
        }
    )
}


#' Matern kernel
#'
#' @name matern_kernel
#' @param sigma Positive number; the parameter.
#' @param l Positive number; the parameter.
#' @param log_nu Real number; the parameter.
#'
#' @return A kernel object.
#'
#' @export
#' 
#' @references # See reference: https://en.wikipedia.org/wiki/Mat%C3%A9rn_covariance_function#Taylor_series_at_zero_and_spectral_moments
matern_kernel <- function(sigma = 1, l = 1, log_nu = log(5)) {
    kernel_template(
        list(sigma = sigma, l = l, log_nu = log_nu),
        function(sigma, l, log_nu) {
            function(X) {
                nu <- exp(log_nu)
                r <- sqrt(X)
                f1 <- sqrt(2 * nu) * r / l
                K <- sigma^2 * (2^(1 - nu) / gamma(nu)) * f1^nu * besselK(f1, nu)
                K[is.na(K)] <- sigma^2
                K
            }
        }
    )
}


l2_norm <- function(x, y) {
    sqrt(sum((x - y)^2))
}
