#
# Code for the paper "Bayesian inference for non-anonymous Growth Incidence Curves"
# written by Edwin Fourrier and Michel Lubrano
#
# Generating a bivariate lognormal
# Chinese equalization example (Exampe 1)
# Bayesian inference

rm(list=ls())

library(compositions)
library(ineq)
library(xtable)
library(mvtnorm)

Bern = function(k,p){
  # Bernstein poly
  # Builds the basis for regression
  z = matrix(NA,n,k+1)
  for (j in 0:k){
    z[,j+1] = choose(k,j)*p^j*(1-p)^(k-j)
  }
  return( z )
}

Bernb = function(b,p){
  # For ploting Bernstein polynomial
  # p is a global variable
  k = length(b)-1
  z = 0
  for (j in 0:k){
    z = z + b[j+1]*choose(k,j)*p^j*(1-p)^(k-j)
  }
  return( z )
}

dBern = function(b,p){
  # First derivative of Bernstein poly
  # p is a global variable
  k = length(b)-1
  z = 0
  for (j in 0:(k-1)){
    z = z+(b[j+2]-b[j+1])*choose(k-1,j)*p^j*(1-p)^(k-j-1)
  }
  return( k*z )
}


##### Example 1 #####

set.seed(1357)
n = 500
r = -0.20
c = 2
S = matrix(c(0.50, r,r,c*0.1),byrow=TRUE,nrow=2)
Y = rlnorm.rplus(n,c(1,1.2),S)
id1 = order(Y[,1])
gna = log(Y[id1,2])-log(Y[id1,1])
y = gna
p = seq(1,n)/n

k = 5
z = Bern(k,p)
out5 = lm(y~0+z) 
summary(out5)
xtable(summary(out5))
plot(p,y,type='l',xlab='p',ylab="Growth rate",main="na-GIC smoothed with a Bernstein polynomial")
ys = predict(out5,se.fit = TRUE,interval = "confidence",level = 0.95)
ci = 2
lines(p,ys$fit[,1],col=ci,lwd=2)
#lines(x,ys$fit[,2],col=ci,lty=2)
#lines(x,ys$fit[,3],col=ci,lty=2)

k = 20
z = Bern(k,p)
out25 = lm(y~0+z) 
summary(out25)
xtable(summary(out25))
ys = predict(out25,se.fit = TRUE,interval = "confidence",level = 0.95)
ci = 3
lines(p,ys$fit[,1],col=ci,lwd=2)
#lines(x,ys$fit[,2],col=ci,lty=2)
#lines(x,ys$fit[,3],col=ci,lty=2)

legend("topright",inset=0.02,legend=c("Natural estimator","Berstein k=5","Bernstein k=20"),
       col=c(1,2,3,4),lty=1)


plot(p,dBern(out5$coefficients,p),type='l',
     ylim=c(-10,4),xlab='p',ylab='Derivative',main="First derivarive of the Bernstein polynomial",
     col=2,lwd=2)
abline(h=0,col=1,lty=2)
lines(p,dBern(out25$coefficients,p),col=3,lwd=2)
legend('top',inset=0.02,legend=c('Derivative k=5','Derivative k=20'),
       col=c(2,3),lty=c(1,1),cex=0.85)



##### End of Program #####
