#
# Code for the paper "Bayesian inference for non-anonymous Growth Incidence Curves"
# written by Edwin Fourrier and Michel Lubrano
#
# Generating a bivariate lognormal
# Chinese equalization example
# Bayesian inference

rm(list=ls())

library(compositions)
library(ineq)
library(xtable)
library(mvtnorm)

Bern = function(k,p){
  # Bernstein poly
  # Builds the basis for regression
  z = matrix(NA,n,k+1)
  for (j in 0:k){
    z[,j+1] = choose(k,j)*p^j*(1-p)^(k-j)
  }
  return( z )
}

Bernb = function(b,p){
  # For ploting Bernstein polynomial
  # p is a global variable
  k = length(b)-1
  z = 0
  for (j in 0:k){
    z = z + b[j+1]*choose(k,j)*p^j*(1-p)^(k-j)
  }
  return( z )
}

dBern = function(b,p){
  # First derivative of Bernstein poly
  # p is a global variable
  k = length(b)-1
  z = 0
  for (j in 0:(k-1)){
    z = z+(b[j+2]-b[j+1])*choose(k-1,j)*p^j*(1-p)^(k-j-1)
  }
  return( k*z )
}

Bregs = function(y,Z,m,b0=0,M0=0,s0=0,nu0=0){
  # Bayesian inference regression with a diffuse prior
  # Return as a list for m draws of beta and sigma2
  
  if (sum(M0)==0){
    kz = ncol(Z)
    M0 = matrix(0,kz,kz)
    b0 = matrix(0,kz,1)
    s0 = 0
    nu0 = 0
  }
  n = length(y)
  M = t(Z)%*%Z + M0
  M1 = solve(M)
  Zy = t(Z)%*%y
  b = M1%*%(M0%*%b0 + Zy)
  se = y%*%y - t(b)%*%M%*%b + s0 + t(b0)%*%M0%*%b0
  s = as.numeric(se)
  nu = n+nu0
  
  Bs = rmvt(m, delta = b, sigma = M1*s/(nu-2), df = nu)
  Ss = s/rchisq(m,n)
  
  return(list(Bs=Bs,Ss=Ss))
}

BDic = function(y,p,ki,km,m,tit,g0=0){
  # Search for Bayesian DIC
  Bc = NULL
  Ac = NULL
  Dc = NULL
  Ml = NULL # Marginal likelihood
  Sig = NULL
  n = length(y)
  for (k in ki:km){
    Z = Bern(k,p)
    b0 = matrix(0,k+1,1)
    ZZ = t(Z)%*%Z
    M0 = g0*ZZ
    M0[,c(1,k+1)] = 0
    M0[c(1,k+1),] = 0
    nu0 = 3
    s0 = 1 
    
    if (g0==0) {
      M0 = matrix(0,k+1,k+1)
      s0 = 0
      nu0 = 0
    }  
    out = Bregs(y,Z,m,b0,M0,s0,nu0)
    Bs = out$Bs
    Ss = out$Ss
    b = colMeans(Bs)
    sb = apply(Bs,2,sd)
    s2 = mean(Ss)
    yZ = y-Z%*%b
    lpy = -n/2*log(s2) - 0.5/s2*t(yZ)%*%yZ
    Bc = cbind(Bc,-2*lpy + (k+1)*log(n))
    Ac = cbind(Ac,-2*lpy + 2*(k+1))
    d0 = log(det(M0[2:k,2:k]))/2
    ml = d0-log(det(M0+ZZ))/2 + (nu0/2)*log(s0)-((n+nu0)/2)*log(s2)
          +(n/2)*log(pi)+lgamma((n+nu0)/2)-lgamma(nu0/2)
    Ml = cbind(Ml,ml)
    Sig = cbind(Sig,s2)
    cat("Posterior results for k = ",k,"\n")
    print(b,digits=3)
    print(sb,digits=3)
    print(b/sb,digits=3)
    print(s2,digits=5)
    cat("\n\n")
      
    lpy = NULL
    for (j in 1:m){
      yZb = y-Z%*%Bs[j,]
      yZbb = t(yZb)%*%yZb
      lpy = cbind(lpy,-n/2*log(Ss[j]) - 0.5/Ss[j]*yZbb)
    }
    Dc = cbind(Dc,-2*mean(lpy)+2*(k+1))
    
  }
  
  cat("Search between",ki,km,"with g0 = ",g0,"\n", 
      " BIC ",which.min(Bc)+ki-1,
      " Marg Lik",which.max(Ml)+ki-1,
      " AIC", which.min(Ac)+ki-1,
      " DIC ",which.min(Dc)+ki-1,
      "\n")
  
  sdx = function(x){
    return((x-min(x))/(max(x)-min(x)))
  }
  pts = function(Dc,ic){
    i = which.min(Dc)
    points(i+ki-1,Dc[i],pch=19,col=ic,cex=1.5)
  }
  Ac = sdx(Ac)
  Bc = sdx(Bc)
  Dc = sdx(Dc)
  Ml = 1-sdx(Ml)
  plot(seq(ki,km),Ac,type='l',xlab = "Polynomial of order k ",ylab = '',main=tit)
  lines(seq(ki,km),Bc,col=2)
  lines(seq(ki,km),Dc,col=3)
  lines(seq(ki,km),Ml,col=4)
  pts(Ac,1)
  pts(Bc,2)
  pts(Dc,3)
  pts(Ml,4)
  legend("top",inset=0.02,legend=c("Marg Lik","BIC","DIC","AIC"),
         col = c(4,2,3,1),lty=1,cex=0.75)

  }


##### Model comparison using Zellner's prior #####

set.seed(1357)
n = 500
r = -0.2
S = matrix(c(0.5, r,r,0.2),byrow=TRUE,nrow=2)
Y = rlnorm.rplus(n,c(1.0,1.2),S)
id1 = order(Y[,1])
p = seq(1,n)/n
gna = log(Y[id1,2])-log(Y[id1,1])
y = gna
ki = 3   # Minimun for k, Bernstein degree
km = 10  # Maximum for k
m = 500

g0 = 0.01
tit = paste("Model comparison with g0 = ",g0)
out = BDic(y,p,ki,km,m,tit,g0)

g0 = 0.1
tit = paste("Model comparison with g0 = ",g0)
out = BDic(y,p,ki,km,m,tit,g0)

g0 = 0.5
tit = paste("Model comparison with g0 = ",g0)
out = BDic(y,p,ki,km,m,tit,g0)

g0 = 1.0
tit = paste("Model comparison with g0 = ",g0)
out = BDic(y,p,ki,km,m,tit,g0)


###### End of Program #####
