#
# Code for the paper "Bayesian inference for non-anonymous Growth Incidence Curves"
# written by Edwin Fourrier and Michel Lubrano
#
# Figs for Bernstein polynomial
# Figure 1: Bernstein and Splines
# Monte Carlo experiment

rm(list = ls())

library(splines)
library(xtable)
library(compositions) # for bivariate lognormal

Bern = function(k,p){
  # Bernstein poly
  # Builds the basis for regression
  z = matrix(NA,n,k+1)
  for (j in 0:k){
    z[,j+1] = choose(k,j)*p^j*(1-p)^(k-j)
  }
  return( z )
}

##### Figure 1 #####

n = 150
x = seq(0, 1, length=n)

k = 10
ks = k+1
Bs = bs(x, degree=3, df=ks,intercept = TRUE)
plot(NA,xlim=c(0,1),ylim=c(0,1),xlab='p',ylab='Cubic splines',
     main= paste("Cubic B-splines with df = ",ks))
for (j in 1:ks){
  lines(x,Bs[,j],col=j)
}

Z = matrix(NA,n,k+1)
x = seq(0,1,length=n)
plot(NA,xlim=c(0,1),ylim=c(0,1),xlab='p',ylab='Berstein',main=paste("Bernstein with k = ",k))
for (j in 0:k){
  Z[,j+1] = choose(k,j)*x^j*(1-x)^(k-j)
  lines(x,Z[,j+1],col=j+1)
}



# Regression MC experiment
# Has to be run several times with n = c(100,1000) and with k = c(3,5,10)


m = 1000
set.seed(13579)
Tab = matrix(NA,m,9)
r = -0.20
S = matrix(c(0.5, r,r,0.1),byrow=TRUE,nrow=2)
mu = c(1.0,1.2)

n = 100
p = seq(0,1,length=n)
ids1 = which(p<=0.05)
ids2 = which(p>=0.95)
k = 5
ks = k+1
Z = Bern(k,p)
Bs = bs(p, degree=3, df=ks,intercept = TRUE)
Ns = ns(p, df=ks,intercept = TRUE)

for (i in 1:m){
  Y = rlnorm.rplus(n,mu,S)
  id1 = order(Y[,1])
  #id2 = order(Y[,2])
  y1 = Y[id1,1]
  y2 = Y[id1,2]
  y = log(y2) - log(y1)
  out = lm(y~0+Z)
  summary(out)
  Tab[i,1] = summary(out)$sigma
  Tab[i,4] = sqrt(mean(out$residuals[ids1]^2))
  Tab[i,7] = sqrt(mean(out$residuals[ids2]^2))
  
  out_Bs = lm(y~0+Bs)
  summary(out_Bs)
  Tab[i,2] = summary(out_Bs)$sigma
  Tab[i,5] = sqrt(mean(out_Bs$residuals[ids1]^2))
  Tab[i,8] = sqrt(mean(out_Bs$residuals[ids2]^2))
  
  out_Ns = lm(y~0+Ns)
  summary(out_Ns)
  res = out_Ns$residuals
  Tab[i,3] = summary(out_Ns)$sigma
  Tab[i,6] = sqrt(mean(res[ids1]^2))
  Tab[i,9] = sqrt(mean(res[ids2]^2))
}
Tb = colMeans(Tab)
Tbc = rbind(Tb[1:3],Tb[4:6],Tb[7:9])
xtable(data.frame(Tbc),digits=3)

Tb = apply(Tab,2,sd)
Tbc = rbind(Tb[1:3],Tb[4:6],Tb[7:9])
xtable(data.frame(Tbc),digits=3)


###### END of program #####
