################################################################################
###--------- Replication of Hauzenberger, Huber, and Koop (2023) ------------### 
###---- Dynamic Shrinkage Priors for TVP Regressions using Scalable MCMC ----###
###------------------ VAR equation-by-equation estimation -------------------###
################################################################################
rm(list = ls())

w.dir <- "TVP_FAST/"
w.dir <- ""
###-------------------- Preliminaries ---------------------------------------###
source(paste0(w.dir, "main_mcmc_sampler.R"))
source(paste0(w.dir, "aux_mcmc_functions.R"))

###-------------- Specifications for grid -----------------------------------###
data.set <- "EA_ylds"   # Yield curve data
freq <- freq.ea <- 12   # Monthly frequency
p <- 2                  # No. of lags
SV   <- TRUE            # Stochastic volatility
cons <- TRUE            # Intercept
rw.st <- TRUE           # RW state equation versus WN state equation
approx.type <- "savs"   # Type of approximation used "savs", "none" or "thrsh"
evol_error  <- "svol-z" # Evolution of state innovation variances: "svol-z", "svol-n", "Mix", "MS" or "static"

# MCMC setup
nburn <- 2500 # No. of burn-in draws
nsave <- 2500 # No. of stored draws
thin  <- 4    # Thinning factor

###----------------------------------- Grid set-up --------------------------###
var.set.ea <- c("S_L", "S_S", "S_C") # Nelson-Siegel model 
shrt.smpl <- 2019 + 11/12  #shortest estimation sample
lng.smpl  <- 2019 + 11/12  #longest estimation sample
end.in <- time(ts(0, start = shrt.smpl, end = lng.smpl, frequency = freq))
  
combi.full <- expand.grid(data.set = "EA_ylds", prior.type = "HS", tvp = TRUE, rw.st = rw.st, approx.type = approx.type, evol_error = evol_error, SV = SV, var.set = var.set.ea, end.in = end.in, stringsAsFactors = FALSE)
run <- 1

###------------------ Extract model specification ---------------------------###
slct.run <- combi.full[run,]

tvp <- combi.full[run, "tvp"] 
rw.st <- combi.full[run, "rw.st"] 
approx.type <- combi.full[run, "approx.type"] 
SV <- combi.full[run, "SV"] 
evol_error <- combi.full[run, "evol_error"]
prior.type <- combi.full[run, "prior.type"]

###---------------- Extract data specification ------------------------------###
data.set <- combi.full[run,"data.set"]
info.set <- substr(combi.full[run,"var.set"],1,1)
target.var <- substr(combi.full[run, "var.set"], 3,100)
end.in <- combi.full[run, "end.in"]
var.set <- substr(var.set.ea,3,100)

load(paste0(w.dir, "EA_ylds.rda"))

Xraw <- apply(ylds[,-1], 2, diff)
Xraw <- apply(Xraw, 2, function(x){(x-mean(x))/sd(x)})
Xraw <- ts(Xraw, end = c(2020,1), frequency = freq.ea)

###---------.--------- Obtain Nelson-Siegel factors -------------------------###
maturities <- c(1:30)*12
lambda <- 0.0609
Lambda <- matrix(0,length(maturities),3) #Three Nelson-Siegel factors
for (jj in 1:length(maturities)){
  l11 <- (1-exp(-maturities[[jj]]*lambda))/(maturities[[jj]] * lambda)
  Lambda[jj,] <- c(1, l11, l11-exp(-maturities[[jj]]*lambda))
}
  
# PCA factors for Nelson Siegel
NS.fac <- ts(matrix(NA, nrow(Xraw), 3), start = time(Xraw)[1], frequency = 12)
  
for(tt in 1:nrow(Xraw))
{
  NS.fac[tt,] <- t(solve(crossprod(Lambda))%*%crossprod(Lambda,Xraw[tt,]))
}
colnames(NS.fac) <- c("L", "S", "C")
ts.plot(NS.fac, type = "l")
Xraw <- NS.fac

# Covariances
if(target.var != var.set[[1]]){
  contemp.var <- 1:(which(var.set == target.var) -1)
}else{
  contemp.var <- NULL
}

Xraw <- Xraw[,var.set]
Xraw <- window(Xraw, end = end.in)
Yraw <- NULL

###--------------- Create specificiation and prior list ---------------------###
M <- ncol(Xraw)
T <- nrow(Xraw)
K <- M*p
K.v <- length(contemp.var)

specification.list <- list()

# Variable specification for triangularization
specification.list$target.var <- target.var
specification.list$contemp.var <- contemp.var
specification.list$var.set <- var.set

# TIV versus TVP 
specification.list$tvp <- tvp
specification.list$rw.st <- rw.st
specification.list$evol_error <- evol_error 
specification.list$prior.type <- prior.type
specification.list$SV <- SV

if(rw.st) off.set.max <- 1/K else off.set.max <- T/K
approx.list <- list(approx.type = approx.type, thrs = 1e-1, zeta = 1/nrow(Xraw), nu = 2, off.set.V0 = 1e-7, off.set.max = off.set.max, train.ind = 0.5, tvp.perc = 0.02)
specification.list$approx.list.list <- approx.list

###---------------------------- Prior setup ---------------------------------###
pr.cnst <- c(0, 10)
names(pr.cnst) <- c("b0", "V0")
ms.prior.list <- list(tau0sq = 1e-1/K^2, tau1sq = 100/K^2, p.prior = 0.1)
ng.prior.list <- list(a.tau.cons = 0.1, a.tau = 0.6, a_i = 0.01, b_i = 0.01, sample_a = FALSE)
Bvarsigma.h <- 1 

hyperprior.list <- list(
  ms.prior.list = ms.prior.list,
  ng.prior.list = ng.prior.list, 
  pr.cnst = pr.cnst, 
  Bvarsigma.h = Bvarsigma.h) 

###------------------------ Directory setup ---------------------------------###
dir.create(paste0(w.dir, data.set), showWarnings = FALSE)
dir <- paste0(w.dir, data.set, "/", info.set)
dir.create(dir, showWarnings = FALSE) 
foldername <- paste0(dir, "/", paste0(slct.run[2:9], collapse = "_"), ".rda")

Regr.obj <- tvp.ds.func(Yraw = Yraw, Xraw = Xraw, p = p, cons = cons, thin = thin, nsave = nsave, nburn = nburn, specification.list = specification.list, hyperprior.list = hyperprior.list)

list2env(x = Regr.obj, envir = .GlobalEnv)
  
k.i <- ncol(Xraw)
if(KKK >= 1){
    A0.i.t.sps_store <- array((-1)*beta.sps.store[,,(KK+1):(KK+KKK),drop =F], c(nsave,T, KKK))
    A0.i.sps_store   <- matrix((-1)*A.sps.store[,(KK+1):(KK+KKK),drop =F],nsave, KKK)
    A0.i.t_store     <- array((-1)*beta.store[,,(KK+1):(KK+KKK),drop =F], c(nsave,T,KKK))
    A0.i_store       <- matrix((-1)*A.store[,(KK+1):(KK+KKK),drop =F],nsave, KKK)
    indA0.i.store    <- array(ind.store[,,(KK+1):(KK+KKK),drop =F], c(nsave,T,KKK))
    dimnames(A0.i.t_store) <- dimnames(A0.i.t.sps_store) <- dimnames(indA0.i.store) <- list(NULL, NULL, var.set[contemp.var])
    colnames(A0.i_store) <- colnames(A0.i.sps_store) <- var.set[contemp.var]
}else{
    A0.i.t.sps_store <- A0.i.t_store <-A0.i_store <- A0.i.sps_store <-  indA0.i.store <- NULL
}
  
if(cons){
    A.i.t.sps_store <- beta.sps.store[,,c(1:KK,K)]
    A.i.sps_store   <- A.sps.store[,c(1:KK,K)]
    
    A.i.t_store <- beta.store[,,c(1:KK,K)]
    A.i_store   <- A.store[,c(1:KK,K)]
    
    indA.i.store <- ind.store[,,c(1:KK,K)]
}else{
    A.i.t.sps_store <- beta.sps.store[,,c(1:KK)]
    A.i.sps_store   <- A.sps.store[,c(1:KK)]
    
    A.i.t_store  <- beta.store[,,c(1:KK)]
    A.i_store    <- A.store[,c(1:KK)]
    
    indA.i.store <- ind.store[,,c(1:KK)]
} 
  
res.str.list <- list(A.i_store = A.i_store, A.i.sps_store = A.i.sps_store, A0.i_store = A0.i_store, A0.i.sps_store = A0.i.sps_store, A.i.t_store = A.i.t_store, A.i.t.sps_store = A.i.t.sps_store, A0.i.t_store = A0.i.t_store, A0.i.t.sps_store = A0.i.t.sps_store, sigma2.store = sigma2.store, fit.store = fit.store, log.lambda.store = log.lambda.store, indA0.i.store = indA0.i.store, indA.i.store = indA.i.store)
spec.list <- list(contemp.var = contemp.var, M=M, K = K, KK = KK, KKK=KKK, T = T, X = X, y =y,Xraw = Xraw, NS.loadings = Lambda)
save(file = foldername, list = c("res.str.list", "spec.list", "nsave"))  