################################################################################
###--------- Replication of Hauzenberger, Huber, and Koop (2023) ------------### 
###---- Dynamic Shrinkage Priors for TVP Regressions using Scalable MCMC ----###
###------------------- Collect individual VAR equations ---------------------###
################################################################################
rm(list = ls())

w.dir <- "TVP_FAST/"
w.dir <- ""

###----------------------- Packages and functions ---------------------------###
library(mvtnorm)
library(MASS)
library(ggplot2)
library(dplyr)
library(reshape2)
library(scales)

get.post <- function(store,dims){
  mean <- apply(store,2:dims,median)
  lo <- apply(store,2:dims,quantile,0.16)
  hi <- apply(store,2:dims,quantile,0.84)
  lo1 <- apply(store,2:dims,quantile,0.05)
  hi1 <- apply(store,2:dims,quantile,0.95)
  return(list(mean,lo,hi,lo1,hi1))
}   


###-------------- Specifications for grid -----------------------------------###
data.set <- "EA_ylds" # Yield curve data
freq <- freq.ea <- 12 # Monthly frequency
p <- 2                # No. of lags
info.set <- "S"       # Information set

SV   <- TRUE            # Stochastic volatility
cons <- TRUE            # Intercept
rw.st <- TRUE           # RW state equation versus WN state equation
approx.type <- "savs"   # Type of approximation used "savs", "none" or "thrsh"
evol_error  <- "svol-z" # Evolution of state innovation variances: "svol-z", "svol-n", "Mix", "MS" or "static"
nsave <- 2500           # No. of stored draws

###----------------------------------- Grid set-up --------------------------###
var.set.ea <- c("S_L", "S_S", "S_C") # Nelson-Siegel model 
shrt.smpl <- 2019 + 11/12  #shortest estimation sample
lng.smpl  <- 2019 + 11/12  #longest estimation sample
end.in <- time(ts(0, start = shrt.smpl, end = lng.smpl, frequency = freq))

combi.full <- expand.grid(data.set = "EA_ylds", info.set = info.set, prior.type = "HS", tvp = TRUE, rw.st = rw.st, approx.type = approx.type, evol_error = evol_error, SV = SV, end.in = end.in, stringsAsFactors = FALSE)
run <- 1
###------------------ Extract model specification ---------------------------###
slct.run <- combi.full[run,]

data.set <- combi.full[run,"data.set"]
info.set <- combi.full[run,"info.set"]
end.in <- combi.full[run, "end.in"]

var.set <- substr(var.set.ea,3,100)
load(paste0(w.dir, "EA_ylds.rda"))
  
Xraw <- apply(ylds[,-1], 2, diff) 
Xraw <- apply(Xraw, 2, function(x){(x-mean(x))/sd(x)})
Xraw <- ts(Xraw, end = c(2020,1), frequency = freq.ea)
Xraw.full <- window(Xraw, end = end.in)
  
###---------.--------- Obtain Nelson-Siegel factors -------------------------###
maturities <- c(1:30)*12
lambda <- 0.0609
Lambda <- matrix(0,length(maturities),3) #three Nelson-Siegel factors
for (jj in 1:length(maturities)){
  l11 <- (1-exp(-maturities[[jj]]*lambda))/(maturities[[jj]] * lambda)
  Lambda[jj,] <- c(1, l11, l11-exp(-maturities[[jj]]*lambda))
}
# PCA factors for Nelson Siegel
NS.fac <- ts(matrix(NA, nrow(Xraw), 3), start = time(Xraw)[1], frequency = 12)
for(tt in 1:nrow(Xraw))
{
  NS.fac[tt,] <- t(solve(crossprod(Lambda))%*%crossprod(Lambda,Xraw[tt,]))
}

###------------------------ Directory setup ---------------------------------###
dir <- paste0(w.dir, data.set, "/", info.set)
foldername <- paste0(dir, "/", paste0(c(slct.run[3:8], info.set, var.set[1], slct.run[9]), collapse = "_"), ".rda")
load(foldername)
list2env(spec.list,globalenv())
X <- X
Y <- Xraw[-c(1:p),]
M <- ncol(Xraw)

A_store <- array(NA, c(nsave, KK+cons, M))
A0_store <- array(NA, c(nsave, M, M))
A.t_store <- array(0, c(nsave, T, KK+cons, M))
A0.t_store <- array(NA, c(nsave, T, M, M))
for(jj in 1:M) A0.t_store[,,jj,jj] <- 1
Sig.t.store <-  array(NA, c(nsave, T, M, M))
indA_mean <- array(NA, c(T, KK+cons, M))
indA0_mean <- array(NA, c(T, M, M))
dimnames(indA_mean)  <- list(time(Xraw)[-c(1:p)], c(paste0(rep(colnames(Xraw), p), "_", rep(paste0("t-", 1:p), each = ncol(Xraw))), "cons"), paste0(colnames(Xraw), "_", rep("t", each = ncol(Xraw))))
dimnames(indA0_mean) <- list(time(Xraw)[-c(1:p)], paste0(colnames(Xraw), "_", rep("t", each = ncol(Xraw))), paste0(colnames(Xraw), "_", rep("t", each = ncol(Xraw))))
  
for (ii in 1:length(var.set)){
target.var <- var.set[ii]  
foldername <- paste0(dir, "/", paste0(c(slct.run[3:8], info.set, target.var, slct.run[9]), collapse = "_"), ".rda")
  
if(is(try(load(foldername), silent = T), "try-error")){
foldername2 <- paste0(dir, "/", paste0(c("ERROR", slct.run[3:8], info.set, slct.run[9]), collapse = "_"), ".rda")
save(file = foldername2, list = c("ii", "target.var"))
stop("Equation missing.")
}

load(foldername)
list2env(res.str.list,globalenv())
k.i <- ncol(Xraw)
  
A_store[,,ii] <- A.i_store 
A.t_store[,,,ii] <- A.i.t_store
indA_mean[,,ii] <- apply(indA.i.store, c(2,3), mean)
if(ii > 1){
    A0_store[,ii,1:(ii-1)] <- A0.i_store
    A0.t_store[,,ii,1:(ii-1)] <- A0.i.t_store
    indA0_mean[,ii,1:(ii-1)] <- apply(indA0.i.store, c(2,3), mean)
}
Sig.t.store[,,ii,ii] <- sigma2.store
}

indA_plot <- melt(indA_mean, varnames = names(indA_mean))
indA0_plot <- melt(indA0_mean, varnames = names(indA_mean))
colnames(indA_plot)  <-  c("T", "K", "M", "Values")
colnames(indA0_plot) <-  c("T", "M", "K", "Values")

indAA0_plot <- rbind(indA_plot, indA0_plot)
indAA0_plot$M <- factor(indAA0_plot$M, levels = c("L_t", "S_t", "C_t"), labels = c("$L_{t}$", "$S_{t}$", "$C_{t}$"))
appender <- function(x) latex2exp::TeX(x)

l_col <- "white"
h_col <- "firebrick1"
h_col <- "firebrick3"

pdf(file = paste0(dir, "/", "coeffs_covs-", paste0(slct.run[c(3,5:8)], collapse = "-"), ".pdf"), height = 6.5, width = 10)
print(ggplot(indAA0_plot, aes(x=K, y=T)) +
        geom_tile(aes(fill=Values), color= alpha("grey", 0.1), size=0.01) + 
        #geom_tile() + geom_raster()  +
  scale_y_reverse(breaks= c(2005:2019, 2019+11/12), labels= c(paste0(2005:2019, ":01"), "2019:12")) +
  scale_x_discrete(breaks= c("L_t-1", "S_t-1", "C_t-1", "L_t-2", "S_t-2", "C_t-2", "cons", "L_t", "S_t", "C_t"), labels= c(latex2exp::TeX("$L_{t-1}$"), latex2exp::TeX("$S_{t-1}$"), latex2exp::TeX("$C_{t-1}$"), latex2exp::TeX("$L_{t-2}$"), latex2exp::TeX("$S_{t-2}$"), latex2exp::TeX("$C_{t-2}$"), latex2exp::TeX("$ic$"), latex2exp::TeX("$L_{t}$"), latex2exp::TeX("$S_{t}$"), latex2exp::TeX("$C_{t}$"))) +
  facet_wrap(~M, labeller = as_labeller(appender, default = label_parsed)) + 
  ylab("") + xlab("") + 
  #scale_fill_gradient(low= l_col, high=h_col, name= "prob.", na.value = "grey80", limits = c(0,1)) +
  scale_fill_gradientn(colours=c(l_col, alpha(h_col, 0.5), alpha(h_col, 0.75), h_col, "black"),
                         values= c(0, 0.3, 0.6, 0.9, 1),name= "PIP", na.value = "grey90",limits = c(0,1),
                         guide="colorbar") +  
  theme_bw() + theme(axis.text.x = element_text(angle = 45, vjust = 1,size = 12, hjust = 1), panel.border = element_rect(colour = "black", fill=NA, size=0.5), panel.grid.major = element_blank(), panel.grid.minor = element_blank(), axis.title.y=element_blank(), strip.background = element_rect(colour="white", fill="grey99"), legend.text=element_text(size=14), strip.text.x = element_text(size = 14), legend.title=element_text(size=14))) 
dev.off()

