% Empirical Study: The cay model
% CUSUM test of quantile cointegration.
% This code is an adaptation of the cusum_quantile_interest.m that was 
% orinally created by Nina Kuriyama. For further on this, please refer to 
% https://www.degruyter.com/document/doi/10.1515/snde-2013-0107/html

clear all;

data = xlsread('Data_Quineche_2021.xlsx',1); % skip first row because var names.

y= data(:, [1]);
x=[data(:, [2]) data(:, [3])]; 


%% Compute the quantile regression coefficients and residuals

q =[0.1 0.20 0.30 0.40 0.50 0.60 0.70 0.80 0.90];  % nine quantiles
Lq=length(q);

[T N]=size(x);
xlag=[zeros(1,N);x(1:T-1,:)];

v=x-xlag;
vlag=[zeros(1,N); v(1:T-1,:)];

for n=1:N
    [rho_2,se_v,t_v]=ols(v(:,n), [ones(T,1) vlag(:,n)]);  % for calculation of the kernel estimator later
    rho_v(n,1)=rho_2(2,1);  % rho_v is N-by-1
    res_v=v(:,n)-[ones(T,1) vlag(:,n)]*rho_2;  % ols residuals
    ss_v(n,1)=res_v'*res_v/(T-1);  % s^2=SSR/(n-k), ss_v is N-by-1
end

beta0=zeros(N,1);

D_T_tail=[1 zeros(1,N); zeros(N,1) eye(N)*sqrt(T)];

[theta_ols,se_ols,t_ols]=ols(y,[ones(size(y)) x]);
u_hat=y-[ones(size(y)) x]*theta_ols;

for k=1:Lq

    [theta,se,t]=qreg(y,[ones(size(y)) x], q(k));
    alpha = theta(1,1);
    beta = theta(2:(N+1),1);
    se_beta=se(2:(N+1),1);
    u_tau=y-[ones(size(y)) x]*theta;
    psi=q(k)-(u_tau<0);
    
    % Compute the density funtion
    F_inv_tau=quantile(u_hat,q(k));
    M_f=1.364*((2*sqrt(pi))^(-1/5))*sqrt(var(u_hat))*(T^(-1/5));  % bandwidth (Silverman's rule of thumb)
    f_hat=sum(normpdf((F_inv_tau-u_hat)/M_f))/(T*M_f);  % using Gaussian kernel to calculate the density function

    % Choose kernel function and bandwidth to compute the fully-modified estimator
    psilag=[0; psi(1:T-1)];
    [rho_1,se_psi,t_psi]=ols(psi, [ones(size(psilag)) psilag]);  % for kernel estimator
    rho_psi=rho_1(2,1);
    res_psi=psi-[ones(size(psilag)) psilag]*rho_1;	% residuals
    ss_psi=res_psi'*res_psi/(T-1); 	% scalar s^2=SSR/(n-k)
    
   %% Using the Bartlett kernel function:
    a1=4*(rho_psi^2*ss_psi/(((1-rho_psi)^6)*((1+rho_psi)^2))+sum(((rho_v.^2).*ss_v)./(((1-rho_v).^6).*((1+rho_v).^2))))/(ss_psi/((1-rho_psi)^4)+sum(ss_v./((1-rho_v).^4)));  % for Bartlett kernel
    M=2*T^(1/3);  % bandwidth
    M=ceil(M);
    h_set=0:M;
    K=bartlett(h_set/M);  % Bartlett kernel function, row vector

    psi_th_1=zeros(T,M);
    v_th_1=zeros(T,M,N);
    v_th=zeros(T,M+1,N);
    for h=1:M
        psi_th_1(:,h)=[psi((h+1):T);zeros(h,1)];  % column vector
        for n=1:N
            v_th_1(:,h,n)=[v((h+1):T,n);zeros(h,1)];
        end
    end

    K_1=K(1,2:M+1);

    psi_th=[psi psi_th_1];
    Delta_psipsi=K*(psi'*psi_th)'/T;  % for (omega_psi.v)^2
    Lambda_psipsi_prim=K_1*(psi'*psi_th_1)'/T;  % for (omega_psi.v)^2

    Delta_vpsi=(K*(v'*psi_th)')'/T;  % N-by-1

    for n=1:N
        v_th(:,:,n)=[v(:,n) v_th_1(:,:,n)]; 
        Delta_vv(:,n)=(K*(v'*v_th(:,:,n))')'/T;  % N-by-N  
        Lambda_vpsi_prim(n,1)=K_1*(psi'*v_th_1(:,:,n))'/T;  % Lambda_vpsi_prim is N-by-1
        Lambda_vv_prim(n,:)=K_1*(v'*v_th_1(:,:,n))'/T;  % Lambda_vv_prim is N-by-N
    end

    Omega_vpsi=Delta_vpsi+Lambda_vpsi_prim;
    Omega_vv=Delta_vv+Lambda_vv_prim;

    omega2_psi=Delta_psipsi+Lambda_psipsi_prim;  % for (omega_psi.v)^2
    Omega_psiv=Omega_vpsi';  % for (omega_psi.v)^2, which is 1-by-N

    Delta_vpsi_plus=Delta_vpsi-Delta_vv*inv(Omega_vv)*Omega_vpsi;

    omega2_psiv=omega2_psi-Omega_psiv*inv(Omega_vv)*Omega_vpsi;  % for (omega_psi.v)^2

    beta_fm=beta-inv((x-ones(size(y))*mean(x))'*(x-ones(size(y))*mean(x))*f_hat)*((x-ones(size(y))*mean(x))'*v*inv(Omega_vv)*Omega_vpsi+T*Delta_vpsi_plus);  % beta_fm is N-by-1

    beta_set_fm(:,k)=beta_fm;
    

    se_beta_fm=diag(inv(sqrt((x-ones(size(y))*mean(x))'*(x-ones(size(y))*mean(x)))*f_hat)*sqrt(omega2_psiv));
    t_beta_fm=(beta_fm-beta0)./se_beta_fm;
    t_beta_set_fm(:,k)=t_beta_fm;
    
    alpha_set_fm(:,k)=alpha; 

    theta_fm=[alpha;beta_fm];

    % CUSUM cointegration test:
 
    y_plus=y-(Omega_psiv*inv(Omega_vv)*v')';  % fm residual
    u_tau_plus=y_plus-[ones(size(y_plus)) x]*theta_fm;  % fm residual
    psi_plus=q(k)-(u_tau_plus<0);  % psi of fm residual
    
    cs=max(abs(cumsum(psi_plus))/sqrt(T*omega2_psiv));  % CS test statistic of the CUSUM test
    cs_set(1,k)=cs;

end

% The critical values are from Hao+Inder (1996) Table 1 (# of regressors=1, excl. constant):
% cv[10% 5% 1%]= [1.0477 1.1684 1.4255]

% The critical values are from Hao+Inder (1996) Table 1 (# of regressors=2, excl. constant):
% cv[10% 5% 1%]= [0.9342  1.0413 1.2561]


rej_cointeg_hi1_05=abs(cs_set)>1.1684;  % 1-by-9, 5% level
% rej_cointeg_hi1_05=abs(cs_set)>1.0413;  % 1-by-9, 5% level
rej_cointeg_hi1_01=abs(cs_set)>1.4255;  % 1-by-9, 1% level
%rej_cointeg_hi1_01=abs(cs_set)>1.2561;  % 1-by-9, 1% level

% Display the result: fully modified slope coefficient estimator, test statistic, and inference 
% where 1 denotes significance at the level considered and 0 denotes insignificance. 

table=[alpha_set_fm; beta_set_fm; cs_set; rej_cointeg_hi1_05; rej_cointeg_hi1_01]  

