function [LL,LLS,ht] = stfigarch_likelihood(parameters,p,q,epsilon,epsilon2,truncLag,errorType,estimFlag)
% Log likelihood for STFIGARCH(Q,D,P) estimation
%
% USAGE:
%   [LL, LLS, HT] = stfigarch_likelihood(PARAMETERS, P, Q, EPSILON, EPSILON2, TRUNCLAG, ERRORTYPE, ESTIMFLAG)
%
% INPUTS:
%   PARAMETERS - A vector of FIGARCH process parameters
%                   [omega phi d beta1 beta2 [nu lambda]]'
%   EPSILON    - T by 1 Column vector of mean zero residuals
%   EPSILON2   - TRUNCLAG + T by 1 column vector containing TRUNCLAG backcasts followed by
%                  EPSILON.^2.  That is [zeros(TRUNCLAG,1)+BACKCAST;EPSILON.^2]
%   P          - 0 or 1 indicating whether the autoregressive term is present in the model (phi)
%   Q          - 0 or 1 indicating whether the moving average term is present in the model (beta)
%   ERRORTYPE  - The type of error being assumed, valid types are:
%                     1 if 'NORMAL'
%                     2 if 'STUDENTST'
%                     3 if 'GED'
%                     4 if 'SKEWT'
%   ESTIMFLAG  - [OPTIONAL] Flag (0 or 1) to indicate if the function is being used in estimation.
%                  If it is 1, then the parameters are transformed from unconstrained values to
%                  constrained by standard garch model constraints
%
% OUTPUTS:
%   LL             - Minus 1 times the log likelihood
%   LLS            - Time series of log likelihoods (Also multiplied by -1)
%   HT             - Time series of conditional variances
%
% COMMENTS:
%  See also FIGARCH, FIGARCH_PARAMETER_CHECK, FIGARCH_WEIGHTS
%  FIGARCH_STARTING_VALUES, FIGARCH_TRANSFORM, FIGARCH_ITRANSFORM



if nargin==8 && estimFlag
    [parameters,nu,lambda] = stfigarch_itransform(parameters,p,q,errorType);
end

stfigarchWeightParameters = parameters(2:3+p+q);
cp = parameters(p+q+4);
gammap = parameters(p+q+5);
%G = 1/(1+exp(-gammap*(epsilon(1)-cp)));
omega = parameters(1);
T = size(epsilon,1);
%archWeights = zeros(truncLag,T);
%G = zeros(T,1);
%for j=1:T
%    G(j) = 1/(1+exp(-gammap*(epsilon(j)-cp)));
%    archWeights(:,j) = stfigarch_weights(stfigarchWeightParameters,p,q,truncLag,G(j));
%end
lag = 1;

G = 1./(1+exp(-gammap.*(epsilon - cp)));
G = [zeros(truncLag,1);G];
tau = truncLag+1:truncLag+T;
ht = zeros(size(epsilon2));
for t = tau;
    archWeights = stfigarch_weights(stfigarchWeightParameters,p,q,truncLag,G(t-lag));
    ht(t) = omega + archWeights'*epsilon2(t-1:-1:t-truncLag);
end
ht = ht(tau);

%Compute the log likelihoods
switch errorType
    case 1
        [LL, LLS] = normloglik(epsilon,0,ht);
        LLS = -LLS;
        LL = -LL;
    case 2
        [LL, LLS] = stdtloglik(epsilon,0,ht,nu);
        LLS = -LLS;
        LL = -LL;
    case 3
        [LL, LLS] = gedloglik(epsilon,0,ht,nu);
        LLS = -LLS;
        LL = -LL;
    case 4
        [LL, LLS] = skewtloglik(epsilon,0,ht,nu,lambda);
        LLS = -LLS;
        LL = -LL;
end

