function [trans_parameters,nu,lambda] = stfiaparch_transform(parameters,p,q,errorType)
% STFIAPARCH(Q,D,P) parameter transformation.  Used to map parameters from a STFIAPARCH process to the
% real line. Used in the estimation of STFIAPARCH. 
%
% USAGE:
%   [TRANS_PARAMETERS,NU,LAMBDA]=stfiaparch_transform(PARAMETERS,P,Q,ERROR_TYPE)
%
% INPUTS:
%   PARAMETERS       - Column parameter vector
%   P                - 0 or 1 indicating whether the autoregressive term is present in the model (phi)
%   Q                - 0 or 1 indicating whether the moving average term is present in the model (beta)
%   ERROR_TYPE       - One of:
%                        1 - Gaussian Innovations
%                        2 - T-distributed errors
%                        3 - Generalized Error Distribution
%                        4 - Skewed T distribution
%
% OUTPUTS:
%   TRANS_PARAMETERS - A 7+p+q column vector of transformed parameters corresponding to
%                      [omega phi d beta1 beta2 gamma delta theta c]' 
%   NU               - Distribution kurtosis parameter, empty if not applicable
%   LAMBDA           - Distribution asymmetry parameter, empty if not applicable
%
% COMMENTS:
%   Input parameters must satisfy:
%    (1) omega > 0
%    (2) 0<= d <= 1
%    (3) 0 <= phi <= (1-d)/2 
%    (3) 0 <= beta <= d + phi 
%    (5) -1< gamma < 1
%    (6) delta > 0
%    (7) nu>2 of Students T and nu>1 for GED
%    (8) -.99<lambda<.99 for Skewed T
%
% See also STFIAPARCH
 
 
 
 
%Handle nu, >2.01 in the case of a T, 1.01<nu<50 in the case of a GED
%T uses square, ged uses a logistic
if errorType==2 || errorType==4
    nu=parameters(p+q+8);
    nu=sqrt(nu-2.01);
elseif errorType==3
    nu=parameters(p+q+8);
    temp=(nu-1)/49;
    nu=log(temp/(1-temp));
else
    nu = [];
end
 
%Lambda must be between -.99 and .99.  Use a logistic
if errorType==4
    lambda=parameters(p+q+9);
    temp=(lambda+0.995)/1.99;
    lambda=log(temp/(1-temp));
else
    lambda = [];
end
 
 
% 0<d<1-2*phi
% 0<beta<phi+d
% omega >0
omegaTrans  = log(parameters(1));
% Find d
if p
    d = parameters(3);
else
    d = parameters(2);
end
% 0<d<1
dTrans = log(d/(1-d));
% phi < (1-d)/2
if p
    phi = parameters(2);
    phiTrans = phi/((1-d)/2);
    phiTrans = log(phiTrans/(1-phiTrans));
    phiplusd = phi + d;
else
    phiTrans = [];
    phiplusd = d;
end
if q
    beta1 = parameters(3+p);
    beta1Trans = beta1 / phiplusd;
    %0 <  beta < phi + d
    beta1Trans = log(beta1Trans/(1-beta1Trans));
    beta2 = parameters(4+p);
    beta2Trans = beta2 / phiplusd;
    %0 <  beta < phi + d
    beta2Trans = log(beta2Trans/(1-beta2Trans));
    gammaTrans  = (parameters(p+5)+1)/2;
    gammaTrans  = log(gammaTrans./(1-gammaTrans));
    deltaTrans  = (parameters(p+6)-.3)/3.7; 
    deltaTrans  = log(deltaTrans./(1-deltaTrans));
    thetaTrans = log(parameters(p+7));    
    cTrans = parameters(p+8);
else
    beta1Trans = [];
    beta2Trans = [];
    cTrans = [];
    thetaTrans = [];
    gammaTrans  = [];
    deltaTrans  = [];
end

trans_parameters= [omegaTrans;phiTrans;dTrans;beta1Trans;beta2Trans;gammaTrans;deltaTrans;thetaTrans;cTrans];

