function [startingvals,nu,lambda,LLs,outputParameters]=stfiaparch_starting_values(startingvals,epsilon,epsilon2,p,q,errorType,truncLag)
% Perform a grid search to find starting values for STFIAPARCH(Q,D,P)
% estimation.  If starting values are user supplied (and thus nonempty), reformats
% starting values depending on ERRORTYPE.
%
% USAGE:
%   [STARTINGVALS,NU,LAMBDA,LLS,OUTPUTPARAMETERS] = ...
%        stfiaparch_starting_values(STARTINGVALS,EPSILON,EPSILON2AUGMENTED,P,Q,ERRORTYPE,TRUNCLAG)
%
% INPUTS:
%   STARTINGVALS     - A vector of starting values or empty to perform a grid search
%   EPSILON          - Vector of mean zero residuals
%   EPSILON2         - TRUNCLAG + T by 1 column vector containing TRUNCLAG backcasts followed by
%                        abs(EPSILON).  That is [zeros(TRUNCLAG,1)+BACKCAST;abs(EPSILON)]
%   P                - 0 or 1 indicating whether the autoregressive term is present in the model (phi)
%   Q                - 0 or 1 indicating whether the moving average term is present in the model (beta)
%   ERRORTYPE        - The type of error being assumed, valid types are:
%                        1 if 'NORMAL'
%                        2 if 'STUDENTST'
%                        3 if 'GED'
%                        4 if 'SKEWT'
%   TRUNCLAG
%
% OUTPUTS:
%   STARTINGVALS     - A vector of starting values (1+p+o+q) by 1
%   NU               - Distribution kurtosis parameter, empty if not applicable
%   LAMBDA           - Distribution asymmetry parameter, empty if not applicable
%   LLS              - A vector of log likelihoods corresponding to OUTPUTPARAMETERS
%   OUTPUTPARAMETERS - A matrix of alternative starting values, sorted by log likelihood
%
% COMMENTS:
%   See also FIGARCH
 
 
%Initialize variables
LLs=[];
outputParameters=[];
 
%No starting values provided
if isempty(startingvals)
    nu=[];
    lambda=[];
 
    %Procedure is to find best starting values, using a grid search find values for normal, then
 
    %Possible starting values based on commonly estimated values
    cs = [0 2 4];
    thetas = [1.5 2 4];    
    ds = [.2 .5 .7];
    phiRatio = [.2 .5 .8];
    betaRatio = [.05 .25 .450];
    betaRatiostar = [.05 .25 .450];
    gammas = [-0.4 0.5 .8];
    deltas = [2.1 2.5 2.4];
    
    covar = cov(epsilon);
    % Construct all of the parameters
    ld=length(ds);
    lp=length(phiRatio);
    lq=length(betaRatio);
    outputParameters=zeros(ld,7+p+q);
    index = 1;
    for i=1:lp
        for j=1:lq
            for k=1:ld
                d = ds(k);
                c = cs(k);
                theta = thetas(k);
                gamma = gammas(k);
                delta = deltas(k);
                if p
                    phi = (1-d)/2 * phiRatio(i);
                else
                    phi = 0;
                end
                beta1 = (d + phi) * betaRatio(j);
                beta2 = (d + phi) * betaRatiostar(j);
                if p && q
                    temp = [phi d beta1 beta2];
                elseif p
                    temp = [phi d];
                elseif q
                    temp = [d beta1 beta2];
                else
                    temp = d;
                end
                G = 1/(1+exp(-theta*(mean(epsilon)-c)));
                lambda = stfiaparch_weights(temp,p,q,truncLag,G);
                omega = covar * (1-sum(lambda));
                outputParameters(index,:)  = [omega temp gamma delta theta c];
                index = index + 1;
            end
        end
    end
    
    outputParameters = unique(outputParameters,'rows');
    %Many outputParameters and LLs
    LLs=zeros(size(outputParameters,1),1);
 
    %Use an index to count
    for i=1:size(outputParameters,1);
        LLs(i) = stfiaparch_likelihood(outputParameters(i,:),p,q,epsilon,epsilon2,truncLag,1,false);
    end
    
    
    %Sort the LLs so the best (lowest, since we minimize the -1*LL)
    [LLs,index]=sort(LLs);
    %Order the starting values
    startingvals=outputParameters(index(1),:)';
    %Order the ouputParameters
    outputParameters=outputParameters(index,:);
    %Use generic values for nu and lambda if needed
    if errorType==2
        nu=8;
    elseif errorType==3
        nu=1.9;
    elseif errorType==4
        nu=8;
        lambda=-.1;
    end
else
    %Values provided, only parse them
    nu=[];
    lambda=[];
    if errorType==2 || errorType==3 || errorType==4
        nu=startingvals(p+o+q+8);
    end
    if errorType==4
        lambda=startingvals(p+o+q+9);
    end
    startingvals=startingvals(1:p+o+q+7);
end


