function [trans_parameters, trans_garch, tP] = MSfigarch_transformation(parameters, k)

% swgarch(k) parameter transformation.  Used to map parameters from a
% swgarch process to the positive unit simplex. Used in the unconstrained estimation of
% swgarch.
%
% USAGE:
%   [trans_parameters, trans_garch, tP]=swgarch_trans(parameters, k, ~,ms_type)
%
% INPUTS:
%   parameters       - Column parameter vector
%   k                - Positive scalar integer representing the number of
%                      regimes
%   error_type       - The type of error being assumed, valid types are:
%                        1 if 'NORMAL'
%                        2 if 'STUDENTST' [NOT IMPLEMENTED YET]  
%   
%   ms_type          - 1 for 'GRAY'     [NOT IMPLEMENTED YET]
%                    - 2 for 'KLAASSEN'
%                    - 3 for 'HAAS'
%
% OUTPUTS:
%   trans_parameters - A 3 x k + (k x k) column vector of transformed parameters corresponding to
%                      [omega_1, ...,omega_k ,alpha_1, ... , alpha_k, beta_1, ... ,beta_k]'
%   trans_garch      - A 3 x k column vector of transformed GARCH parameters
%   tP               - A k x k column vector of transformed transition probabiliies 
%                      parameters
%   
%
% COMMENTS:
%   Input parameters must satisfy:
%    (1) omega_j > 0
%    (2) alpha_j >= 0 for j = 1,...,k
%    (3) beta_j  >= 0 for j = 1,...,k
%    (4) A stationary process (more constraints on all the parameters)
%    (5) The p_ij sum in colum to one 
%
% See also swgarch
%
% Copyright: Thomas Chuffart
% Mail: thomas.chuffart@univ-amu.fr
% Version: MSG_tool_Beta v2.0   Date: 30/01/2015


% Parse the parameters and if equal to 0, just add 1e-8

omega=parameters(1:k);
phi=parameters(k+1:k*2);
d=parameters(k*2+1:k*3);
beta=parameters(k*3+1:k*4);
beta(beta==0) = 1e-8;
phi(phi==0) = 1e-8;
omega(omega==0) = 1e-8;

% 0<d<1-2*phi
% 0<beta<phi+d
% omega >0
omegaTrans  = log(omega);
% Find d
% 0<d<1
dTrans = log(d./(1-d));
% phi < (1-d)/2
kc = ((1-d)./2);
phiTrans = phi./kc;
phiTrans = log(phiTrans./(1-phiTrans));
phiplusd = phi + d;

betaTrans = beta./phiplusd;
    %0 <  beta < phi + d
betaTrans = log(betaTrans./(1-betaTrans));

% construct the Markovian matrix

P = parameters(4*k+1:4*k+(k*k));
Pstat = reshape(P,k,k)';




% Check that the parameters satisfy the necessary constraints
if  any(phi<0) || any(beta<0);
    error('These do not conform to the necessary set of restrictions to be transformed.')
end

    theta = diag(beta);
    for i = 1:k,
        for j = 1:k,
            e = zeros(1,k);
            e(i) = 1;
            Mtemp  = Pstat(i,j)*(theta+phi*e);
            Mc((i*k)-(k-1):(i*k),(j*k)-(k-1):(j*k)) = Mtemp;
        end    
    end
    if max(abs(eig(Mc))) > 1,
        error('These do not conform to the necessary set of restrictions to be transformed.')
    end

%Initialze the transformed transition probability matrix
Pstat = reshape(Pstat,k,k);
tP = Pstat;

for i = 1:k,
    for j = 1:k,
        tP(j,i) = log(Pstat(i,j)/Pstat(j,j));
    end
end

% Parse the parameters

trans_garch = [omegaTrans;phiTrans;betaTrans];
tP = reshape(tP,k*k,1);

% Merge the parameters
trans_parameters=[omegaTrans;phiTrans;dTrans;betaTrans;tP];

end
