function lambda = fiaparch_weights(parameters,p,q,truncLag)
% FIAPARCH(Q,D,P) ARCH(oo) weight computation forfor P={0,1} and Q={0,1}
%
% USAGE:
%   [LAMBDA] = fiaparch_weights(PARAMETERS,P,Q)
%
% INPUTS:
%   PARAMETERS    - A (1+P+Q) by 1 column vector of parameters in the order [phi d beta] where phi
%                     is omitted if P=0 and beta is omitted if Q=0 
%   P             - 0 or 1 indicating whether the autoregressive term is present in the model (phi)
%   Q             - 0 or 1 indicating whether the moving average term is present in the model (beta)
%   TRUNCLAG      - Number of weights to compute in ARCH(oo) representation
%
% OUTPUTS:
%   PARAMETERS    - TRUNCLAG by 1 vector or weights for FIAPARCH calculation
%
%    The conditional variance, h(t), of a FIAPARCH(1,d,1) process is modeled
%    as follows:
%    which is estimated using an ARCH(oo) representation, 
%
%    h(t) = omega + sum(lambda(i) * z(t-1))
%      z(t-1) = (abs(epsilon2(t-1:-1:t-truncLag))-gamma*epsilon2(t-1:-1:t-truncLag)).^(delta)
%    
%    where lambda(i) is a function of the fractional differencing parameter, phi and beta
%    


% Parse parameters
if q
    beta = parameters(2+p);
else
    beta = 0;
end

if p
    phi = parameters(1);
    d = parameters(2);
else
    phi = 0;
    d = parameters(1);
end


% Recursive weight computation
lambda = zeros(truncLag,1);
deltaa = zeros(truncLag,1);
lambda(1) = phi - beta + d;
deltaa(1) = d;
for i=2:truncLag
    deltaa(i) = (i-1-d)/i * deltaa(i-1);
    lambda(i) = beta*lambda(i-1) + (deltaa(i) -phi *deltaa(i-1));
end