function [LL,LLS,ht] = fiaparch_likelihood(parameters,p,q,data,epsilon2,truncLag,errorType,estimFlag)
% Log likelihood for FIAPARCH(Q,D,P) estimation
%
% USAGE:
%   [LL, LLS, HT] = fiaparch_likelihood(PARAMETERS, P, Q, EPSILON, EPSILON2, TRUNCLAG, ERRORTYPE, ESTIMFLAG)
%
% INPUTS:
%   PARAMETERS - A vector of FIAPARCH process parameters
%                   [omega phi d beta gamma delta [nu lambda]]'
%   EPSILON    - T by 1 Column vector of mean zero residuals
%   EPSILON2   - TRUNCLAG + T by 1 column vector containing TRUNCLAG backcasts followed by
%                  EPSILON.^2.  That is [zeros(TRUNCLAG,1)+BACKCAST;EPSILON.^2]
%   P          - 0 or 1 indicating whether the autoregressive term is present in the model (phi)
%   Q          - 0 or 1 indicating whether the moving average term is present in the model (beta)
%   ERRORTYPE  - The type of error being assumed, valid types are:
%                     1 if 'NORMAL'
%                     2 if 'STUDENTST'
%                     3 if 'GED'
%                     4 if 'SKEWT'
%   ESTIMFLAG  - [OPTIONAL] Flag (0 or 1) to indicate if the function is being used in estimation.
%                  If it is 1, then the parameters are transformed from unconstrained values to
%                  constrained by standard garch model constraints
%
% OUTPUTS:
%   LL             - Minus 1 times the log likelihood
%   LLS            - Time series of log likelihoods (Also multiplied by -1)
%   HT             - Time series of conditional variances
%

if nargin==8 && estimFlag
    [parameters,nu,lambda] = fiaparch_itransform(parameters,p,q,errorType);
end

omega = parameters(1);
gamma = parameters(p+q+3);
delta = parameters(p+q+4);
deltainv=2/delta;
fiaparchWeightParameters = parameters(2:2+p+q);
T = size(data,1);
archWeights = fiaparch_weights(fiaparchWeightParameters,p,q,truncLag);
tau = truncLag+1:truncLag+T;
ht = zeros(size(epsilon2));
for t = tau;
    ht(t) = omega + archWeights'*(abs(epsilon2(t-1:-1:t-truncLag))-gamma*epsilon2(t-1:-1:t-truncLag)).^(delta);
end
ht = ht(tau).^(deltainv);

%Compute the log likelihoods
switch errorType
    case 1
        [LL, LLS] = normloglik(data,0,ht);
        LLS = -LLS;
        LL = -LL;
    case 2
        [LL, LLS] = stdtloglik(data,0,ht,nu);
        LLS = -LLS;
        LL = -LL;
    case 3
        [LL, LLS] = gedloglik(data,0,ht,nu);
        LLS = -LLS;
        LL = -LL;
    case 4
        [LL, LLS] = skewtloglik(data,0,ht,nu,lambda);
        LLS = -LLS;
        LL = -LL;
end