%***********************************************************************************************/
%*                                                                                             */
%*    Modified Classical Cycle Dating program                                                  */
%*                                                                                             */
%*    > Computes turning points as in Harding and Pagan (2002) and a recession index           */
%*    > Computes a concordance index and a correlation coefficient across cycles as well       */
%*    > Transforms the recession index in order to produce a recession bar (in excel)          */
%*                                                                                             */
%*    The turning points are determined by the following rule (which can be modified):         */
%*        peak : must be greater than 'turnphase' (2 quarters for real GDP) on either side     */
%*        trough : must be smaller than 'turnphase' (2 quarters for real GDP) on either side   */
%*    A turning point must meet some additional criteria (restrictions):                       */
%*        Each 'phase' (contraction/expansion) lasts at least 2 quarters for real GDP          */
%*        Each full 'cycle' phase (contraction & expansion) lasts at least 4 quarters          */
%*              for real GDP                                                                   */
%*        Troughs and peaks must alternate : if there are two peaks (troughs) in a row,        */
%*              it will select the highest (lowest)                                            */
%*        Each trough must be lower than the preceeding peak                                   */
%*        There are no turning points within 'phase' length (2 quarters for real GDP)          */
%*              end points                                                                     */
%*                                                                                             */
%*                                                                                             */
%*    Date: August 1, 2018                                                                     */
%*    Author: Enrique Martinez-Garcia (Federal Reserve Bank of Dallas)                         */
%*    Email: emg.economics@gmail.com                                                           */
%*       modified from James Engel's version of the Adrian Pagan and Don Harding's BBQ code    */
%*       James Engel's original code at: http://www.ncer.edu.au/data/                          */
%*                                                                                             */
%***********************************************************************************************/

% -----------------------------------------------------------------------------------------------
% NOTE: Aspects of the code to keep in mind when using it :
%
% 1. Cycle characteristics can be updated ('phase', 'turnphase, and/or 'cycle') if the cyclical 
% properties of the series or the series themselves change. However, if the number of series 
% included or the length of the time series changes, STEP 1.a on data loading and STEP 7 on 
% data saving would not have to be modified. The code automatically adjusts them to the new 
% range in the excel file (RGDPG8_cycles.xlsx).
%
% 2. This code has been written for data series that are of either monthly or quarterly 
% frequency.
%
% 3. The code works only with balanced panels. Hence, if the original data is not balanced, a 
% correction would be applied to ensure a balanced panel. Only the time series span for which 
% all series are complete will be included.
% -----------------------------------------------------------------------------------------------

tic;
close all;
clear all;
clc;

% Cycle characteristics - NEEDS TO BE MODIFIED DEPENDING ON THE FREQUENCY OF THE DATA USED
frequency = 2;          % Frequency of the data : 1 - months, 2 - quarters
turnphase = 2;          % Determines the rule for selecting a local max/min as occuring when the value in the period is greater/smaller than 2 quarters/5 months on either side for real PDI and real GDP, 3 quarters on real house prices
phase = 2;              % Censoring rule : determines the minimum length of a contraction or expansion on real PDI and real GDP (2 quarters/6 months), 3 quarters on real house prices
cycle = 4;              % Censoring rule : determines the minimum length of a full cycle on real PDI and real GDP (4 quarters/15 months), 6 quarters on real house prices
thresh = 10.4;          % Bypasses 'phase' and 'cycle' restriction if peak to trough is greater than 'thresh' (can be modified)


% STEP 1.a Data loading 

% Alert that the code is loading the data
%disp(' ');
disp(' The code is loading the data on real GDP (RGDP).')
disp(' ');

% Import the identification labels and the real GDP (RGDP) data
[~, ~, raw1] = xlsread('RGDPG8_cycles.xlsx','DATA');                       % Real GDP (RGDP) data

country = raw1(1:2,2:end);                                                 % Keeps the first two rows (reserved for names/identification) but eliminates the first column (reserved for dates) and all the data
country(cellfun(@(x) ~isempty(x) && isnumeric(x) && isnan(x),country)) = {''};
countrylong = country(1,:);                                                % Long country names
country = country(2,:);                                                    % Short country names

raw1 = raw1(3:end,2:end);                                                  % Eliminates the two first rows (reserved for names) and the first column (reserved for dates)
R1 = cellfun(@(x) ~isnumeric(x) || isnan(x),raw1);                         % Find non-numeric cells
raw1(R1) = {nan};                                                          % Replace non-numeric cells with NaN

% Create output variable - data is loaded in levels, and re-expressed in logs
RGDP = log(cell2mat(raw1));
m=size(RGDP,2);                                                            % Number of series in the dataset for real GDP (determined by the number of columns in the dataset)
nd=size(RGDP,1);                                                           % Number of observations in the sample (determined by the number of rows in the dataset for real GDP)

% Clear temporary variables
clearvars raw1 R1;

% Dataset to be used in the analysis
rawdata = nan(nd,m);
rawdata(1:nd,1:m) = RGDP;    
name = [country']; 


% STEP 1.b Ensuring a balanced panel, detection of gaps within the time series

% This for-loop identifies the location of the first number (ninit) and the last number (nfinal) of each series in the dataset
% ninit = ones(1,m); nfinal = nd*ones(1,m);
for jj=1:m;                  
    ninit(1,jj)  = min(find(~isnan(rawdata(:,jj))));
    nfinal(1,jj) = max(find(~isnan(rawdata(:,jj))));
jj=jj+1;
end;
clearvars jj;

% Warning about the nature of the unbalanced panel and the correction made
if max(ninit) == 1
    if min(nfinal) == nd
        disp(' ');
        disp(' ');
        disp(' This is a balanced panel data where all series have the same lenght. ');
        disp(' ');
    else
        disp(' ');
        disp(' ');
        disp(' WARNING: This is an unbalanced panel data where all series begin in the same period, but for some series observations are missing at the end. ');
        disp(' The incomplete rows at the end have been eliminated from the dataset. Now the panel is balanced. ');
        disp(' ');
    end;
else
   if min(nfinal) == nd
        disp(' ');
        disp(' ');
        disp(' WARNING: This is an unbalanced panel data where all series end in the same period, but for some series observations are missing at the beginning. ');
        disp(' The incomplete rows at the beginning have been eliminated from the dataset. Now the panel is balanced. ');
        disp(' ');
    else
        disp(' ');
        disp(' ');
        disp(' WARNING: This is an unbalanced panel data where all series begin and end at different times. For some series observations are missing at the beginning and/or end. ');
        disp(' The incomplete rows at the beginning and end have been eliminated from the dataset. Now the panel is balanced. ');
        disp(' ');
   end;
end;

pp1=nfinal-(ninit-1);
pp2=nd-sum(isnan(rawdata));
rawdata = rawdata(max(ninit):min(nfinal),:);                        % This makes the necessary correction to the dataset so that the panel will be balanced (where all series overlap)
nd = size(rawdata,1);                                               % This adjusts the size of the dataset to take into account the rows eliminated to balance the panel

% However there may be gaps in the series and we also need to check for those and correct them
% Check if there are gaps in the data and warn about them
pp=pp1-pp2;
datagaps = find(~pp==0);
countrygaps=name([datagaps],:);

if sum(pp)==0
else
    disp(' ')
    disp(' ************************************************************************************************ ')
    disp(' WARNING: There are some series with gaps in-between the data. The countries with such gaps are : ')
    disp('   ')
    for i=1:size(datagaps,2)
        disp(['     ' char(countrygaps(i))])
        i=i+1;
    end
    disp('   ')
    disp(' Review the data reported for those countries in the excel file to eliminate the gaps')
    disp(' in the series appropriately and start the code over again. ')
    disp(' If you have problems with the code, make sure to contact the author at: ')
    disp(' Enrique Martinez-Garcia: enrique.martinez-garcia@dal.frb.org ')
    disp(' ')
    return
end
clearvars pp1 pp2 pp;
    
    
% STEP 1.c Data transformation and sample selection

% Switch: 1-only uses complete cycles, 0-uses incomplete cycles as well
disp(' ');
disp(' Do you want to use only complete cycles (choice = 1) or to consider incomplete cycles as well (choice = 2)? ');
disp(' (Default: choice = 2) ');
disp(' ');
choice1=input(' Choice : ');
if isempty(choice1);choice1=2;end

    if choice1 == 1;
        complete = 1;
    else
        complete = 0;
    end;

% Sample selection for the analysis
disp(' ');
disp(' Do you want use the entire sample in the dataset (choice = 1) or to explore a subsample (choice = 2)? ');
disp(' (Default: choice = 1) ');
disp(' ');
choice2=input(' Choice : ');
if isempty(choice2);choice2=1;end

    if choice2 == 1;
        disp(' ');
    else
        
        disp(' ');
        if frequency==1;
            disp([' There are ' num2str(nd) ' observations in the sample contained in the dataset (monthly frequency).'])
        else
            disp([' There are ' num2str(nd) ' observations in the sample contained in the dataset (quarterly frequency).'])
        end;
        disp(' Do you want to eliminate observations from the beginning of the sample (choice = -X) or from the end of the sample (choice = X)? ');
        disp(' (Default: choice = 0). NOTE: X must be an integer number smaller than the full sample that indicates the number of observations to be droped. ');
        disp(' Example: Cutting seven years of a sample of quarterly data is equivalent to droping 28 observations. ');
        disp(' ');
        choice2a=input(' Number of Observations to Be Excluded : ');
        if isempty(choice2a);choice2a=0;end

            if choice2a == 0;
                disp(' ');
            elseif choice2a <=-1;
                rawdata=rawdata(1-choice2a:nd,:);
                nd=size(rawdata,1);
            else
                rawdata=rawdata(1:nd-choice2a,:);
                nd=size(rawdata,1);
            end;
            
    end;

    
% STEP 2. Calculate the turning points of the series

% Calculate peak to trough average durations and average amplitudes
% p stands for peaks, t for troughs, p gives contractions, t expansions
% the labels pd,td are durations; the labels pda,tda are the amplitudes

% Introduction of matrices and scalars required in the calculations
bcp5=zeros(nd,1);bct5=zeros(nd,1);st=zeros(nd,1);
nbt=0;                                       % Number of troughs
nbp=0;                                       % Number of peaks


i=1;
while i<=m;

% STEP 2.1b Calculate the turning points of the series for complete cycles
    
pdm=0;pdma=0;tdm=0;tdma=0;

[bcp5,bct5,nbp,nbt]=rawall(rawdata(1:nd,i),turnphase,nd,phase,cycle,thresh);     % Calculates turning points with restrictions

if nbp+nbt<=2;

else
 
                ntr=nbt;npk=nbp;

                nr=[nbt;nbp];
                nv=max(nr);

                pdc=zeros(nv,1);tdc=zeros(nv,1);
                pda=zeros(nv,1);tda=zeros(nv,1);td=zeros(nv,1);pd=zeros(nv,1);
                
                if bcp5(1,1) < bct5(1,1);       % Peaks are first 

                    nr=[nbt;nbp];
                    r=nbt;
                      pd=bct5(1:r,1)-bcp5(1:r,1);
                      pda=rawdata(bct5(1:r,1),i)-rawdata(bcp5(1:r,1),i);

                    k=1;
                    while k<=r;
                      pdc(k)=sumc(rawdata(bcp5(k,1):bct5(k,1),i)-rawdata(bcp5(k,1),i));
                    k=k+1;
                    end;
                else;                           % Troughs are first

                r=nbt-1;
                  pd=bct5(2:r+1,1)-bcp5(1:r,1);
                  pda=rawdata(bct5(2:r+1,1),i)-rawdata(bcp5(1:r,1),i);

                k=1;
                while k<=r;
                  pdc(k)=sumc(rawdata(bcp5(k,1):bct5(k+1,1),i)-rawdata(bcp5(k,1),i));         
                k=k+1;
                end;

                r1=r;

                end;

                % Calculate trough to peak durations & amplitudes 

                if bct5(1,1) < bcp5(1,1);        % Troughs are first 
                r=nbp;
                  td=bcp5(1:r,1)-bct5(1:r,1);                               
                  tda=rawdata(bcp5(1:r,1),i)-rawdata(bct5(1:r,1),i);                    
                k=1;
                while k<=r;
                  tdc(k)=sumc(rawdata(bct5(k,1):bcp5(k,1),i)-rawdata(bct5(k,1),i));

                k=k+1;
                end;

                else;                           % Peaks are first 
                  r=nbp-1;
                  td=bcp5(2:r+1,1)-bct5(1:r,1);                             
                  tda=rawdata(bcp5(2:r+1,1),i)-rawdata(bct5(1:r,1),i);                  


                k=1;
                while k<=r;

                  tdc(k)=sumc(rawdata(bct5(k,1):bcp5(k+1,1),i)-rawdata(bct5(k,1),i));

                k=k+1;
                end;

                end;
                pdc=pdc(1:rows(pd));
                tdc=tdc(1:rows(td));

% STEP 2.1b Calculate the turning points of the series if incomplete cycles are allowed

% NOTE: allowing incomplete cycles means that if the first turning point is
% a peak (trough), then it treats the first observation of the series as
% a trough (peak) for the purpose of computing the amplitude and duration
% of the cycles. Similarly, if the last turning point is a peak (trough),
% then it treats the last observation of the series as a trough (peak).
                
if complete == 0;                          % Switch: 0-uses incomplete cycles as well

    bct5u=bct5;
    bcp5u=bcp5;
    
        if bcp5(1,1) < bct5(1,1);          % Modifies code to include incomplete cycles
        
        	bct5u  = [1;bct5];
            
        elseif 	bct5(1,1) < bcp5(1,1);
        
        	bcp5u = [1;bcp5]; 
            
        end;
        
        
        nbtu = rows(bct5u);
        nbpu = rows(bcp5u);
        
        if bcp5u(nbpu,1) < bct5u(nbtu,1);    % Modifies code to include incomplete cycles
        
        	bcp5u  = [bcp5u;nd];
            
        elseif 	bct5u(nbt,1) < bcp5u(nbp,1);
        
        	bct5u = [bct5u;nd]; 
            
        end;
        
        
       ntr=rows(bct5u);npk=rows(bcp5u);
       nr=[ntr;npk];
       nv=max(nr);

        
        pdc=zeros(nv,1);tdc=zeros(nv,1);
        pda=zeros(nv,1);tda=zeros(nv,1);td=zeros(nv,1);pd=zeros(nv,1);
        pdc=zeros(nv,1);tdc=zeros(nv,1);
        
        
         if bcp5u(1,1) < bct5u(1,1);         % Peaks are first 

                    nr=[ntr;npk];
                    r=ntr;
                      pd=bct5u(1:r,1)-bcp5u(1:r,1);
                      pda=rawdata(bct5u(1:r,1),i)-rawdata(bcp5u(1:r,1),i);
                        k=1;
                        while k<=r;
                         pdc(k)=sumc(rawdata(bcp5u(k,1):bct5u(k,1),i)-rawdata(bcp5u(k,1),i));
                        k=k+1;
                        end;
                else;                         % Troughs are first
                r=ntr-1;
                  pd=bct5u(2:r+1,1)-bcp5u(1:r,1);
                  pda=rawdata(bct5u(2:r+1,1),i)-rawdata(bcp5u(1:r,1),i);
                    k=1;
                    while k<=r;
                      pdc(k)=sumc(rawdata(bcp5u(k,1):bct5u(k+1,1),i)-rawdata(bcp5u(k,1),i));         
                    k=k+1;
                    end;

                r1=r;

                end;

                % Calculate trough to peak durations & amplitudes 

                if bct5u(1,1) < bcp5u(1,1);      % Troughs are first 

                r=npk;
                  td=bcp5u(1:r,1)-bct5u(1:r,1);                             
                  tda=rawdata(bcp5u(1:r,1),i)-rawdata(bct5u(1:r,1),i);                  
                    k=1;
                    while k<=r;
                      tdc(k)=sumc(rawdata(bct5u(k,1):bcp5u(k,1),i)-rawdata(bct5u(k,1),i));

                    k=k+1;
                    end;


                else;                            % Peaks are first 
                  r=npk-1;
                  td=bcp5u(2:r+1,1)-bct5u(1:r,1);                           
                  tda=rawdata(bcp5u(2:r+1,1),i)-rawdata(bct5u(1:r,1),i);                


                    k=1;
                    while k<=r;

                      tdc(k)=sumc(rawdata(bct5u(k,1):bcp5u(k+1,1),i)-rawdata(bct5u(k,1),i));

                    k=k+1;
                    end;

                end;
                pdc=pdc(1:rows(pd));
                tdc=tdc(1:rows(td));
        
 end;                        
                
              
% STEP 2.2 Calculate the duration and amplitude of contractions and expansions

                pdm=pdm+meanc(pd);          % Duration of contractions
                pdma=pdma+meanc(pda);       % Amplitude of contractions

                tdm=tdm+meanc(td);          % Duration of expansions
                tdma=tdma+meanc(tda);       % Amplitude of expansions
                    
               
end;

% STEP 2.3 Compute the peaks and throughs

        disp('   ')
        disp([' Statistics on Average Cycle for ' char(name(i,:))])

% Number of peaks and number of troughs 
disp([' ' num2str(nbp) ' Peaks At Periods :'])
disp(bcp5(1:nbp)')

disp([' ' num2str(nbt) ' Troughs At Periods :'])
disp(bct5(1:nbt)')


% Determine observation range in which states have been completed -- computes a contraction index (cont_index)
[st]=states(bcp5,bct5,nbp,nbt,nd);

na=min([bct5(1);bcp5(1)])';
nb=max([bct5(nbt);bcp5(nbp)])';
nb-na+1;

cont_index(:,i)= st;


% STEP 3. Report the characteristics of the cycle for each series

if nbp+nbt>2;
    
        if frequency==1;
            disp(' Average Duration in Months :')
            disp(' Contractions / Expansions')
            disp([pdm tdm]) 
        else
            disp(' Average Duration in Quarters :')
            disp(' Contractions / Expansions')
            disp([pdm tdm]) 
        end;
        
        disp(' Average Amplitude (in Percentages) :')
        disp(' Contractions / Expansions')
        disp([pdma*100 tdma*100])

        else
    
        disp(' ')
        disp([' Calculations for Average Duration and amplitude are skipped since the number of turning points (peaks and troughs) is only 2 or less for '  char(name(i,:))])
        disp(' ')

end;

     i=i+1;
end;


if m>=2;
    
% STEP 4. Compute the concordance index

cc=zeros(m,m);

for j=1:m;
    for i=(1+j):m;
        cc(j,i)=nanmean(cont_index(:,j).*cont_index(:,i)+(1-cont_index(:,j)).*(1-cont_index(:,i)));
    end;
end;

% STEP 5. Compute the GMM estimator for the correlation coefficient

for j=1:m;
    for i=(1+j):m;
        cc(i,j)=nanmean(((cont_index(:,j)-nanmean(cont_index(:,j))).*(cont_index(:,i)-nanmean(cont_index(:,i))))/((nanmean(cont_index(:,j)).*(1-nanmean(cont_index(:,j))).*nanmean(cont_index(:,i)).*(1-nanmean(cont_index(:,i))))^0.5));
    end;
end;

% Loop to produce a matrix for the concordance index of the series investigated
ccstr=strvcat(char('------'),char(name));
for j=1:m;
    ccstr2=strvcat(char(name(j)),num2str(cc(:,j)));
    ccstr=horzcat(ccstr,ccstr2);
end
clearvars ccstr2;

disp('   ')
disp('   ')
disp(' Matrix of the concordance index (elements above the diagonal) and the correlations (elements below the diagonal) :')
disp(' ')
disp([ccstr])   

else
    disp('   ')
    disp('   ')
    disp(' A concordance index and a correlation coefficient cannot be computed because there are fewer than 2 series in the dataset.')
    disp('   ')

end;


% STEP 6 (Optional). Report the matrix that categorizes the phases of contraction and expansions

disp(' ');
disp(' Do you want to display the index of contraction periods (marked with a 0) and expansion periods (marked with a 1) ');
disp(' for each series in the dataset (choice = 1) or to end the program now (choice = 2)? (Default: choice = 1) ');
disp(' ');
choice3=input(' Choice : ');
if isempty(choice3);choice3=1;end

    if choice3 == 1;
        % Loop to produce the index of contraction periods (indicated with a zero) for the series investigated
        cont_index_table=strvcat(char(name(1)),num2str(cont_index(:,1)));
        for j=2:m;
            cont_index_table2=strvcat(char(name(j)),num2str(cont_index(:,j)));
            cont_index_table=horzcat(cont_index_table,cont_index_table2);
        end
        clearvars cont_index_table2;
        disp(' ');
        disp(' Indicator function for cyclical expansions and contractions :')
        disp([cont_index_table])
        disp(' ');
    else
        disp(' ');
    end;


% STEP 7. The construction of contraction bar indicators -- Saved as output in the same excel file that contains the data

% The matrix cont_index is a dummy indicator that assigns the value of zero to contractions and the value of one to expansions.
% It has also the property that the last 1 in a series before it turns to 0 represents a peak while the last 0 in a series 
% before it turns to 1 represents a trough.
% The contraction bar indicator will assign a value of 1 from peak to trough and zero otherwise, so the matrix cont_index 
% has to be properly adjusted for that purpose. I call the contraction bar indicator matrix recessions

disp(' ');
disp(' The code is computing and saving an indicator function for cyclical contractions that assigns the value of one from peak to trough.')
disp(' This indicator is the primary output of the code and is being saved in the excel spreadsheet where the original data is stored.')
disp(' ');

% This loop creates the contraction indicator matrix with all the series available
for i=1:m;
    for j=1:nd;
        if isnan(cont_index(j,i))
            recessions(j,i) = cont_index(j,i);
        elseif cont_index(j,i) == 0
            recessions(j,i) = 1;
        else
            if j+1>nd
                recessions(j,i) = 0;
            elseif (j+1<=nd)&(cont_index(j+1,i) == 1)
                recessions(j,i) = 0;
            else
                recessions(j,i) = 1;
            end
        end
    end
end

% Here I create separate recession bar indicators for each variable and store them in the original excel file
% Adjusting the recession bar indicators for the inclusion of real GDP
xlswrite('RGDPG8_cycles.xlsx',recessions,'CYCLES','B3');                   % Saves the real GDP peak-to-trough indicator in an excel file

% For house-keeping: Clear temporary variables
clearvars i1 i j k;



minutes=toc/60;
disp(' ****************************************************************** ')
disp([' The program has ended its run. The time spent has been ' num2str(minutes) ' minutes. '])
disp(' If you are not able to replicate the results presented, ')
disp(' make sure to contact the authors at: ')
disp(' Enrique Martinez-Garcia: emg.economics@gmail.com ')