library(ggplot2)
library(cowplot)
theme_set(theme_cowplot())
library(stringr)
library(tidyr)
library(forcats)
library(lme4)
library(lmerTest)
library(corrplot)

library(dplyr)

read.table("acoustic.txt",T) -> cogs
# reorder factor for plotting; clean
cogs$phoneCode <- factor(cogs$phoneCode, levels=c("s","ɿ","ʮ","ɕ","iʑ","yʑ"))
cogs <- droplevels(cogs[cogs$place != "ignore",])

###

# violin plot of all tokens of data
png("raws-v25.png", width=6, height=3.5, units="in", res = 600)
ggplot(cogs, aes(phoneCode,cog/1000,fill=fct_rev(YoB))) + 
	geom_violin(fill="#00BFC450",color="#00BFC400",alpha=0.25) +
	geom_boxplot(width=0.4) + 
	scale_fill_manual(labels=c("pre-1985", "post-1985"),
		values=c("#F8766D","gray90")
		) + 
	labs(y="CoG (kHz)", x="Phone", fill="Cohort") +
	background_grid(major="y")
dev.off()

###

# lmers

# change the reference levels to: high-CoG (female, apical, unvoi, unrounded)
cogs$place <- relevel(cogs$place, ref="apical")
cogs$voice <- relevel(cogs$voice, ref="unvoi")
cogs$round <- relevel(cogs$round, ref="unrd")
cogs$YoB <- relevel(cogs$YoB, ref="pre-1985")
cogs$gender <- relevel(cogs$gender, ref="female")

# subsetting
m1segs <- c("s","ɿ","ɕ","iʑ")
cogs_m1 <- cogs[cogs$phoneCode %in% m1segs,]
m2segs <- c("ɿ","ʮ","iʑ","yʑ")
cogs_m2 <- cogs[cogs$phoneCode %in% m2segs,]

# the models
m1.g <- lmer(cog ~ place*voice*YoB*gender + (1+place|subj), data=cogs_m1) # DID converge; debating
m2.g <- lmer(cog ~ place*round*YoB*gender + (1+place|subj), data=cogs_m2) # converged, not used

###

# correlation analysis, plots

# generate an object containing medians, for corrplots
meds <- cogs %>% 
	group_by(subj,gender,YoB,phoneCode) %>% 
	summarize(median_cog = median(cog))
		
meds <- meds %>% 
	spread(key=phoneCode, value=median_cog)

# median scatterplots
post_unr <- ggplot(meds, aes(ɕ/1000, iʑ/1000)) + 
	geom_abline(slope=1, intercept=0, lty=2) +
	geom_point(color="gray") + 
	geom_smooth(method="lm") + 
	xlim(3.5,12) + ylim(3.5,12) + 
	xlab("CoG of /ɕ/ (kHz)") + ylab("CoG of /iʑ/ (kHz)") +
	facet_wrap(~fct_rev(YoB))

post_rd <- ggplot(meds, aes(ɕ/1000, yʑ/1000)) + 
	geom_abline(slope=1, intercept=0, lty=2) +
	geom_point(color="gray") + 
	geom_smooth(method="lm") + 
	xlim(3.5,12) + ylim(3.5,12) +
	xlab("CoG of /ɕ/ (kHz)") + ylab("CoG of /yʑ/ (kHz)") +
	facet_wrap(~fct_rev(YoB))

ant_unr <- ggplot(meds, aes(s/1000, ɿ/1000)) + 
	geom_abline(slope=1, intercept=0,lty=2) +
	annotate(geom="text", x=5.5, y=6.1, label="Identity line", angle = 45) +
	geom_point(color="gray") + 
	geom_smooth(method="lm") + 
	xlim(3.5,12) + ylim(3.5,12) +
	xlab("CoG of /s/ (kHz)") + ylab("CoG of /ɿ/ (kHz)") +
	facet_wrap(~fct_rev(YoB))

ant_rd <- ggplot(meds, aes(s/1000, unlist(meds[,6]/1000))) + 
	geom_abline(slope=1, intercept=0,lty=2) +
	geom_point(color="gray") + 
	geom_smooth(method="lm") + 
	xlim(3.5,12) + ylim(3.5,12) + 
	xlab("CoG of /s/ (kHz)") + ylab("CoG of /ʮ/ (kHz)") +
	facet_wrap(~fct_rev(YoB))

png("corrs-fig6-v2.png", width=10, height=6, units="in", res = 600)
plot_grid(ant_unr, ant_rd, post_unr, post_rd, nrow=2, labels=c("s-ɿ","s-ʮ","ɕ-iʑ","ɕ-yʑ"))
dev.off()

# same, but across rounding, within voicing
post_bth <- ggplot(meds, aes(iʑ/1000, yʑ/1000)) + 
	geom_abline(slope=1, intercept=0,lty=2) +
	geom_point(color="gray") + 
	geom_smooth(method="lm") + 
	xlim(3.5,12) + ylim(3.5,12) + 
	xlab("CoG of /iʑ/ (kHz)") + ylab("CoG of /yʑ/ (kHz)") +
	facet_wrap(~YoB)

ant_bth <- ggplot(meds, aes(ɿ/1000, unlist(meds[,6]/1000))) + 
	geom_abline(slope=1, intercept=0,lty=2) +
	geom_point(color="gray") + 
	geom_smooth(method="lm") + 
	xlim(3.5,12) + ylim(3.5,12) + 
	xlab("CoG of /ɿ/ (kHz)") + ylab("CoG of /ʮ/ (kHz)") +
	facet_wrap(~YoB)

png("corrs-fig7-v2.png", width=10, height=3, units="in", res = 600)
plot_grid(ant_bth, post_bth, ncol=2, labels=c("ɿ-ʮ", "iʑ-yʑ"))
dev.off()

###

# correlation tests and correlograms

meds_y <- meds[meds$YoB == "post-1985",]
meds_o <- meds[meds$YoB == "pre-1985",]
meds.y.cor <- cor(meds_y[meds_y$subj != "S35",4:9]) # exclude b/c NA values
meds.o.cor <- cor(meds_o[,4:9])

res.young <- cor.mtest(meds_y[meds_y$subj != "S35",4:9], conf.level=.95)
res.old <- cor.mtest(meds_o[,4:9], conf.level=.95)

png("cog-corr-old-v2.png",width=3, height=3, units="in", res = 600)
corrplot(meds.o.cor, type='upper', order="AOE", col = c("black", "white"), bg = "lightblue", tl.col = "black", p.mat = res.old$p, insig = "label_sig", sig.level = c(.001, .01, .05), pch.cex = .9, pch.col = "red", diag =TRUE)
dev.off()
png("cog-corr-young-v2.png",width=3, height=3, units="in", res = 600)
corrplot(meds.y.cor, type='upper', order="AOE",  col = c("black", "white"), bg = "lightblue", tl.col = "black", p.mat = res.young$p, insig = "label_sig", sig.level = c(.001, .01, .05), pch.cex = .9, pch.col = "red", diag =TRUE)
dev.off()
